<?php
/**
*  
* This is the main IMEXporter class file. It handles all the necessary operations for the
* export and import methods. IMEXporter is intended to be used as an addon script for any
* application that needs a good, reliable and strong mysql import tool. It handles all the 
* import process via MySQL LOAD DATA INFILE so it processes the data very fast to your database.
*
* @author 	Eduardo Pereira
* @date		25/03/2012
* @website	http://www.voindo.eu/imex
*
* @requirements PHP 5.0+ 
*
* @package IMEXporter.class
* 
*/


class IMEXp extends parseCSV {
	
	/**************************************
	**  DEFINE CLASS VARIABLES RESETING  **
	**		 DO NOT CHANGE HERE			 **
	**************************************/
	public 	$DBconn;
	private $_tableHeaders		= "";
	private $_csvHeaders		= "";
	private $_csvNumLines		= "";
	private $_currentFile		= "";

	private $_FullVerifySTATUS	= true;
	private $_FullVerifyERROR	= array();
	
	private $_defaultDBserver;
	private $_defaultDBuser;
	private $_defaultDBpass;
	private $_defaultDBname;
	private $_defaultDBtype;
	
	private $_CFG_language;
	private $_CFG_langfolder;
	
	private $_outputMsgs;
	private $_outputFVbar;
	private $_directTblCrea;
	
	
	function __construct() {
		
		/*****************************************************
		**			CONFIGURATIONS - CHANGE HERE			**
		*****************************************************/
		$this->_CFG_language 	= "en";								// Set desired language, make sure file is created in folder LOCALE.
		$this->_CFG_langfolder 	= dirname(__FILE__) . "/locale/";	// Set absolute path for LOCALE folder. If it's together with this class, leave it as it is default.
		$this->_defaultDBserver	= "localhost";						// Set default database server
		$this->_defaultDBuser 	= "imexporter";						// Set default database username
		$this->_defaultDBpass 	= "imexporter";						// Set default database password
		$this->_defaultDBname 	= "imexporter";						// Set default database name
		$this->_outputMsgs 		= true;								// Set to false if you do not want status messagens outputted to screen
		$this->_outputFVbar 	= true;								// Set to false if you do not want full verify progress bar to appear
		$this->_directTblCrea 	= true;								// Set to false if you do not want to allow table creation on the interactive import
		/*****************************************************
		**				  END CONFIGURATIONS                **
		*****************************************************/
		
		$this->_defaultDBtype 	= "mysql";
		require_once dirname(__FILE__) . '/adodb/adodb-exceptions.inc.php';
		require_once dirname(__FILE__) . '/adodb/adodb.inc.php';
		try {
			$this->DBconn = &ADONewConnection($this->_defaultDBtype);
			$this->DBconn->Connect($this->_defaultDBserver, $this->_defaultDBuser, $this->_defaultDBpass, $this->_defaultDBname);
		} catch (exception $e) { 
			throw new IMEXp_config_error($this->__L("SYS-DBACCDEN"));
			die();
		}
		$this->DBconn->SetFetchMode(ADODB_FETCH_ASSOC);
		$this->DBconn->EXECUTE("SET names 'utf8'");
	}
	
	
	
	
	
	/***********************************************
		FROM THIS POINT IS FUNCTIONS FOR METHODS
	***********************************************/
	
	
	/**
	* Function exportTable()
	*
	* AVAILABLE METHOD
	*
	* Function to process the table to CSV export process
	* It processes the request and calls class the function _processExport
	* 
	* @param	string 		The table name to export
	* @param	string 		The type of export								- Optional - Default is "csvstring"
	* @param	string 		The result filename								- Optional - Default is empty
	* @param	string 		If csv first line should conatin column names	- Optional - Default is "csvstring"
	*
	* @return	string		PHP Headers with the csv file
	*
	*/
	public function exportTable($tablename, $type = "csvstring", $filename = "", $addtitles = true) {
		
		if ($filename == "")
			$filename 	= $tablename . "_" . date("d_m_Y_H_i_s") . ".csv";
		
		$result 	= &$this->DBconn->Execute("SELECT * FROM $tablename");
		
		if ($type == "csvfile") {
			$output = $this->_processExport($result,',',',',false,$addtitles);
			
			header("Cache-Control: must-revalidate, post-check=0, pre-check=0");
			header("Content-Length: " . strlen($output));
			header("Content-type: text/csv");
			header("Content-Disposition: attachment; filename=$filename");
			
			return $output;
			
		} else if ($type == "csvout") {
			$filename = fopen('php://stdout','wb');
			$output = $this->_processExport($result,',',',',true,$addtitles);
			fclose($filename);
			
			return $filename;
			
		} else {
			$output = $this->_processExport($result, ',', ',', false, $addtitles);
			return $output;
		}
		
	}
	
	
	/**
	* Function importFile_Direct()
	*
	* AVAILABLE METHOD
	*
	* This function is called to execute a direct import CSV to Mysql
	* It does some basic checks and tries to direct import the file to the
	* desired table, or, if allowed, creates a table and imports CSV
	* 
	* @param	string 		The file path (absolute path)
	* @param	string 		The table name
	* @param	bool 		The table creation is allowed					- Optional - Default is "false"
	*
	* @return	string		The result of the import or any error.
	*
	*/
	public function importFile_Direct($file, $table, $create = false) {
		$this->start_OutputContainer();
		$this->_tableHeaders = $this->get_ColumnsNames($table);
		
		if ($this->_tableHeaders === false) {
			$this->outputToScreen("error", $this->__L("SYS-TbNotExistCreaD"));
			$status = "";
		} else {
		
			$row 		= 0;
			$handle 	= fopen ($file,"r");
			while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
				if ($row < 1) {
					$num = count ($data);
					for ($c=0; $c < $num; $c++) {
						$this->_csvHeaders[$c] = $data[$c];
					}
				}
				$row++;
			}
			$this->_csvNumLines = $row - 1;
			fclose ($handle);
			if ($this->_tableHeaders === $this->_csvHeaders) {
				$status	= $this->_processDirect("DATALOAD", $file, $table);
			} else {
				// Headers of the CSV do not match the headers of the table
				$this->outputToScreen("error", $this->__L("SYS-HeadsNoMatchNotDo"));
				$status = false;
			}
		}
		return $status;
	}
	
	
	/**
	* Function importFile_Interative()
	*
	* AVAILABLE METHOD
	*
	* This function enables the second method for CSV import process.
	* This method is a interactive process. It does some checks and allows user
	* intearction when need in case CSV and Table column titles do not match.
	* All the interaction is made outputting valid XHTML to the user.
	*
	* This function can also be called VIA AJAX during the process of importFile() method function
	* bacause it handles part of the fully interactive (importFile()) process.
	* 
	* @param	string 		The file path (absolute path)
	* @param	string 		The table name
	* @param	bool 		If CSV has column titles in first row				- Optional - Default is "true"
	* @param	bool 		If full CSV verification is to be made				- Optional - Default is "true"
	* @param	bool 		If is called from AJAX (importFile() method)		- Optional - Default is "false"
	*
	* @return	string		The result of the import, any error or any interaction forms before import process call.
	*
	*/
	public function importFile_Interative($file, $table, $hasHeaders = true, $fullVerify = true, $isAjax = false) {
		
		$create	= true;
		
		if ($isAjax == false)
			$this->start_OutputContainer();
		
		$this->_currentFile		= $file;
		$this->_tableHeaders	= $this->get_ColumnsNames($table);
		
		// Checks if table does not exist (false), otherwise $this->_tableHeaders holds the table column names (headers)
		if ($this->_tableHeaders === false) {
			// Table does not exists, create it?
			if ($this->_directTblCrea === true) {
				$status = $this->Create_Table_XHTML($table, $hasHeaders);
			} else {
				$this->outputToScreen("error", $this->__L("SYS-TbNotExistCreaD")); // Output error, Table does not exists and table creadtion is disabled
				$status = "";
			}
			
		} else {
			// Table exists 
			if ($isAjax == false)
				$this->outputToScreen("success", $this->__L("SYS-TableFound") . " <strong>".$this->__L("SYS-Done!")."</strong>");
			
			// Provided CSV has column titles (headers) ?
			if ($hasHeaders === true) {
				$this->auto($file);
				$rowCount 	= $this->_csvNumLines = count(file($file)) - 1;		// Set number of CSV rows (count total - first row because its headers)
				$colCount 	= $this->processCSV_Titles();						// Set $this->_csvHeaders with CSV column titles (headers)
				
				// Check if table column titles match csv column titles
				if ($this->_tableHeaders === $this->_csvHeaders) {
					if ($isAjax == false)
						$this->outputToScreen("success", $this->__L("SYS-TblCSVmatch") . " <strong>".$this->__L("SYS-Done!")."</strong>");
					
					// Run full CSV validation ?
					if ($fullVerify === true)
						$FVS 	= $this->runFullVerify($table, $rowCount, $isAjax);
					
					if (($fullVerify === true) and ($FVS["nbFieldsError"] === true)) {
						if ($isAjax == false)
							$status = $this->outputErrorOnLine_XHTML($FVS["stoppedatrow"], $isAjax);
						else {
							$status[]	= 1;
							$status[]	= $this->outputErrorOnLine_XHTML($FVS["stoppedatrow"], $isAjax);
						}
						
					} else if (($fullVerify === true) and ($this->_FullVerifySTATUS === false)) {
						if ($isAjax == false)
							$status = $this->outputFullVerifyError_XHTML(false, null, $isAjax);
						else {
							$status[]	= 1;
							$status[]	= $this->outputFullVerifyError_XHTML(false, null, $isAjax);
						}
						
					} else {
						if ($isAjax == false) {
							if ($fullVerify === true)
								$this->outputToScreen("success", $this->__L("SYS-FullChkDone") . " <strong>".$this->__L("SYS-Done!")."</strong>");
							$status	= $this->_processDirect("DATALOAD", $file, $table);
						} else {
							$status	= $this->_processDirect("AJAX", $file, $table);
						}
					}
					
				} else {
					// Header of the CSV do not match the headers of the table, lets do a match between both CSV and Table headers
					$this->outputToScreen("warning", $this->__L("SYS-HeaderNoMatch") . " <strong>" . $this->__L("SYS-ChooseMatch") . "</strong>", "CSVandTableHeaders_Matcher_warning_msg");
					$status = $this->CSVandTableHeaders_Matcher_XHTML($table, $fullVerify, $rowCount);
				}
				
			} else {
				$this->heading = false;
				$this->auto($file);
				
				$rowCount 	= $this->_csvNumLines = count(file($file));			// Set number of CSV rows
				$colCount 	= $this->processCSV_NoTitles();						// Set $this->_csvHeaders with Manual CSV column titles (Column_1, Column_2, etc)
				
				// Lets do a match table headers to CSV content
				$this->outputToScreen("warning", $this->__L("SYS-CSVnoHeads") . " <strong>" . $this->__L("SYS-ChooseMatch") . "</strong>", "TableHeaders_Matcher_warning_msg");
				$status = $this->CSVandTableHeaders_Matcher_XHTML($table, $fullVerify, $rowCount);
			}
			
		}
		
		return $status;
	}
	
	
	/**
	* Function importFile()
	*
	* This is the main function for the CSV import full yinteractive process.
	* It will output the whole XHTML forms for the file upload, questions and matches,
	* full csv validation, and starts the import process. It actually calls another function
	* that creates all the XHTML to be outputted. Reason is all XHTML functions are together.
	* 
	* @param	array 		The tables to be ignored and not given as choice	- Optional - Default is empty
	* @param	bool 		If tabel creation is an option						- Optional - Default is "true"
	*
	* @return	string		PHP Headers with the csv file
	*
	*/
	public function importFile($ignoredTables = "", $allowCreation = true) {
		return $this->importFile_XHTML($ignoredTables, $allowCreation);
	}
	
	
	
	
	
	/***********************************************
		FROM THIS POINT IS PROCESSOR FUNCTIONS
	***********************************************/
	
	
	/**
	* Function _processExport()
	*
	* This function is called by above exportTable after the request is processed
	* It gets the database tabnle content and returns a CSV string to be output as file download
	* 
	*
	* @return	string		CSV file content
	*
	*/
	private function _processExport(&$rs, $sep, $sepreplace, $fp=false ,$addtitles=true, $quote = '"', $escquote = '', $replaceNewLine = ' ') {
		if (!$rs) return '';
		//----------
		// CONSTANTS
		$NEWLINE = "\r\n";
		$BUFLINES = 100;
		$escquotequote = $escquote.$quote;
		$s = '';
		
		if ($addtitles) {
			$fieldTypes = $rs->FieldTypesArray();
			reset($fieldTypes);
			$i = 0;
			while(list(,$o) = each($fieldTypes)) {
			
				$v = ($o) ? $o->name : 'Field'.($i++);
				if ($escquote) $v = str_replace($quote,$escquotequote,$v);
				$v = strip_tags(str_replace("\n", $replaceNewLine, str_replace("\r\n",$replaceNewLine,str_replace($sep,$sepreplace,$v))));
				$elements[] = $v;
				
			}
			$s .= implode($sep, $elements).$NEWLINE;
		}
		$hasNumIndex = isset($rs->fields[0]);
		
		$line = 0;
		$max = $rs->FieldCount();
		
		while (!$rs->EOF) {
			$elements = array();
			$i = 0;
			if ($hasNumIndex) {
				for ($j=0; $j < $max; $j++) {
					$v = $rs->fields[$j];
					if (!is_object($v)) $v = trim($v);
					else $v = 'Object';
					if ($escquote) $v = str_replace($quote,$escquotequote,$v);
					$v = strip_tags(str_replace("\n", $replaceNewLine, str_replace("\r\n",$replaceNewLine,str_replace($sep,$sepreplace,$v))));
					
					if (strpos($v,$sep) !== false || strpos($v,$quote) !== false) $elements[] = "$quote$v$quote";
					else $elements[] = $v;
				}
			} else { // ASSOCIATIVE ARRAY
				foreach($rs->fields as $v) {
					if ($escquote) $v = str_replace($quote,$escquotequote,trim($v));
					$v = strip_tags(str_replace("\n", $replaceNewLine, str_replace("\r\n",$replaceNewLine,str_replace($sep,$sepreplace,$v))));
					
					if (strpos($v,$sep) !== false || strpos($v,$quote) !== false) $elements[] = "$quote$v$quote";
					else $elements[] = $v;
				}
			}
			$s .= implode($sep, $elements).$NEWLINE;
			$rs->MoveNext();
			$line += 1;
			if ($fp && ($line % $BUFLINES) == 0) {
				if ($fp === true) echo $s;
				else fwrite($fp,$s);
				$s = '';
			}
		}
		
		if ($fp) {
			if ($fp === true) echo $s;
			else fwrite($fp,$s);
			$s = '';
		}
		
		return $s;
	}
	
	
	/**
	* Function _processDirect()
	*
	* This function is called by importFile_Direct() and also _processCreateTable()
	* importFile_Interative() also calls this when Table and CSV column titles match.
	*
	* Basically it processes the CSV file and loads it into the mysql table straight 
	* 
	* @param	string 		The type of import (AJAX or DATALOAD). Only important for the output
	* @param	string 		The filename full absolute path
	* @param	string 		The table name to import to
	* @param	bool 		If the CSV has the column titles in the first row		- Optional - Default is "true"
	*
	* @return	string		The result of the import
	*
	*/	
	private function _processDirect($type, $file, $table, $hasHeaders = true) {
		
		if ($hasHeaders == true)
			$ignoreline = "IGNORE 1 LINES";
		else
			$ignoreline = "";
		
		$sql = "LOAD DATA LOCAL INFILE '".addslashes($file)."' REPLACE INTO TABLE $table FIELDS TERMINATED BY ',' OPTIONALLY ENCLOSED BY '" . '"' . "' ESCAPED BY " . "'\\\\'" . " LINES TERMINATED BY '\n' $ignoreline (";
		$i 		= 0;
		foreach ($this->_tableHeaders as $header) {
			if ($i == 0)
				$sql .= $header;
			else
				$sql .= ", " . $header;
			$i++;
		}
		$sql .= ")";
		
		if ($type == "DATALOAD") {
			if ($this->DBconn->Execute($sql) === false) {
				$this->outputToScreen("error", $this->__L("SYS-ErrorDBOper") . " | " . addslashes(__FILE__) . " | #" . __LINE__ . " | " . $this->DBconn->ErrorMsg() ."");
				return true;
			} else {
				$this->outputToScreen("success", $this->__L("SYS-SuccessLoaded") ." ".$this->_csvNumLines." " . $this->__L("SYS-LinesIntoTb") . " ($table). <strong>".$this->__L("SYS-Done!")."</strong>");
				return false;
			}
		}
		
		if ($type == "AJAX") {
			if ($this->DBconn->Execute($sql) === false) {
				$result[0]	= 1;
				$result[1]	= $this->__L("SYS-ErrorDBOper") . " | " . __FILE__ . " | #" . __LINE__ . " | " . $this->DBconn->ErrorMsg();
				return $result;
			} else {
				$result[0]	= 0;
				$result[1]	= mysql_info();
				return $result;
			}
		}
		
	}
	
	
	/**
	* Function _processMatched()
	*
	* This function is called only after the Table and CSV column titles match form
	* has been submitted in only two possible ocasions. importFile_Interative() and importFile() methods
	*
	* Basically it processes the CSV file and processes the desired matches before loading it into the
	* Mysql table. It assigns the values of the CSV to some Mysql variables and then proceses those
	* variables into the table, as per the match form request. Before, if requested it runs the full CSV
	* validation function, to check all CSV values.
	* 
	* @param	int 		The number of columns
	* @param	string 		The table name to import to
	* @param	string 		The filename full absolute path
	* @param	string 		If full CSV validation is to be made "1" YES everything else NO
	* @param	int 		The number of rows in the CSV
	* @param	array 		The array of the table columns from the FORM (FIXED)
	* @param	array 		The array of the CSV columns from the FORM (USER CHOOSEN)
	*
	* @return	string		It return the mysql import status, or any error from the full validation or from the mysql
	*
	*/	
	public function _processMatched($numberOption, $current_table, $current_file, $fullverify, $numberRows, $tblHeaders, $csvHeaders) {
		
		$tableHeaders	= $this->get_ColumnsNames($current_table);
		
		if ($fullverify === "1") {
			$this->auto($current_file);
			$rowCount 	= $this->_csvNumLines = count(file($current_file)) - 1;		// Set number of CSV rows (count total - first row because its headers)
			$colCount 	= $this->processCSV_Titles();								// Set $this->_csvHeaders with CSV column titles (headers)
			$this->_tableHeaders	= $this->get_ColumnsNames($current_table);
			$FVS 	= $this->runFullVerify($current_table, $numberRows, true, $csvHeaders);
		}
		
		
		if (($fullverify === "1") and ($FVS["nbFieldsError"] === true)) {
			$result[0]	= 3;
			$result[1]	= $this->outputErrorOnLine_XHTML($FVS["stoppedatrow"], true);
			
		} else if (($fullverify === "1") and ($this->_FullVerifySTATUS === false)) {		
			$result[0]	= 3;
			$result[1]	= $this->outputFullVerifyError_XHTML(true, $csvHeaders, true);
			
		} else {
			
			$toVars		= "";
			$setCols	= "";
			
			for ($a=0; $a < $numberOption; $a++) {
				if ($a == 0) {
					$toVars	.= "@var$a";
					$setCols .= $tableHeaders[$a] . " = @var" . $csvHeaders[$a];
				} else {
					$toVars	.= ", @var$a";
					$setCols .= ", " . $tableHeaders[$a] . " = @var" . $csvHeaders[$a];
				}
			}
			
			$sql 	= "LOAD DATA LOCAL INFILE '".addslashes($current_file)."' REPLACE INTO TABLE $current_table FIELDS TERMINATED BY ',' OPTIONALLY ENCLOSED BY '" . '"' . "' ESCAPED BY " . "'\\\\'" . " LINES TERMINATED BY '\n' IGNORE 1 LINES ($toVars) SET $setCols";
			
			if ($this->DBconn->Execute($sql) === false) {
				$result[0]	= 1;
				$result[1]	= $this->__L("SYS-ErrorDBOper") . " | " . __FILE__ . " | #" . __LINE__ . " | " . $this->DBconn->ErrorMsg();
			} else {
				$result[0]	= 0;
				$result[1]	= mysql_info();
			}
		}
		return $result;
	}
	
	
	/**
	* Function _processCreateTable()
	*
	* This function is called everytime a table is going to be created.
	* Basically it creates a new table acoording to the csv column titles
	* (even if column titles are not presetn, it creates column_1, column, 2, etc)
	* and after table creation, this function calls the _processDirect() function 
	* to import the CSV to the newly created table.
	* 
	* @param	string 		The table name to create and import to
	* @param	string 		The filename full absolute path
	* @param	string 		If CSV has column titles in the first row. "0" is false, everything else true
	*
	* @return	string		It returns any errors or else the mysql import status
	*/
	public function _processCreateTable($current_table, $current_file, $hasHeaders) {
		
		if ($hasHeaders == "0")
			$this->heading = false;
		
		$this->auto($current_file);
		$rowCount 	= $this->_csvNumLines = count(file($current_file)) - 1;			// Set number of CSV rows (count total - first row because its headers)
		
		if ($hasHeaders == "1") {
			$colCount 	= $this->processCSV_Titles();								// Set $this->_csvHeaders with CSV column titles (headers)
			
		} else {
			$colCount 	= $this->processCSV_NoTitles();								// Set $this->_csvHeaders with CSV column titles Column_1, Column_2, etc
		}
		
		for ($a = 0; $a < $colCount; $a++) {
			$maxlen[$a]	= 0;
		}
		
		foreach ($this->data as $key => $row):
			$c = 0;
			foreach ($row as $value):
				$len	= strlen($value);
				if (@$maxlen[$c] < $len) {
					@$maxlen[$c] = $len;
				}
				$c++;
			endforeach;
		endforeach;
		
		$sqlcols 		= "";
		$colnameError	= 0;
		for ($a = 0; $a < $colCount; $a++) {
			$colname	= $this->_csvHeaders[$a];
			$colmaxlen	= $maxlen[$a] + 10;
			
			if (!preg_match('/^[a-z][a-z\d_]*$/',$colname))
				$colnameError++;
			
			if ($a == ($colCount - 1)) {
				$sqlcols .= "$colname varchar($colmaxlen)";
			} else {
				$sqlcols .= "$colname varchar($colmaxlen), ";
			}
		}
		
		if ($colnameError == 0) {
			$sql 	= "CREATE TABLE $current_table ($sqlcols)";
			if ($this->DBconn->Execute($sql) === false) {
				$result[0]	= 1;
				$result[1]	= $this->__L("SYS-ErrorDBOper") . " | " . __FILE__ . " | #" . __LINE__ . " | " . $this->DBconn->ErrorMsg();
			} else {
				$this->_tableHeaders = $this->get_ColumnsNames($current_table);
				$result	= $this->_processDirect("AJAX", $current_file, $current_table, $this->heading);
			}
			
		} else {
			$result[0]	= 1;
			$result[1]	= $this->__L("SYS-ErrCreatTableCSV");
		}
		
		return $result;
		
	}
	
	
	
	
	
	
	/***********************************************
	   FROM THIS POINT IS AJAX RELATED FUNCTIONS
	***********************************************/
	
	
	/**
	* Function importFile_AJAX_checkHeaders()
	*
	* This function is only called during the importFile() method process
	*
	* It will take the csv file, and checks it against the
	* desired table to see if column and csv headers (column names) match
	* 
	* @param	string 		The file path (absolute path)
	* @param	string 		The table name
	* @param	string 		If CSV has titles in first row ("1" has and "0" does not have)
	*
	* @return	string		One of the possible result strings ("HEADERS_MATCH", "CSV2TABLE_HEADERS_MATCHER", "TABLE_HEADERS_MATCHER")
	*
	*/
	public function importFile_AJAX_checkHeaders($file, $table, $hasHeaders) {
		
		$conclusion	= "";
		
		$this->_currentFile		= $file;
		$this->_tableHeaders	= $this->get_ColumnsNames($table);
		
		// Provided CSV has column titles (headers) ?
		if ($hasHeaders == "1") {
			
			$this->auto($file);
			$this->auto($file);
			// Set $this->_csvHeaders with CSV column titles (headers)
			$this->processCSV_Titles();
			
			if ($this->_tableHeaders === $this->_csvHeaders) {
				// CSV and Table Headers match
				$conclusion	= "HEADERS_MATCH";
			} else {
				// Header of the CSV do not match the headers of the table, lets do a match between both CSV and Table headers
				$conclusion	= "CSV2TABLE_HEADERS_MATCHER";
			}
			
		} else {
			
			$this->heading = false;
			$this->auto($file);
			// Set $this->_csvHeaders with Manual CSV column titles (Column_1, Column_2, etc)
			$this->processCSV_NoTitles();
			// Lets do a match table headers to CSV content
			$conclusion	= "TABLE_HEADERS_MATCHER";
		}
		
		return $conclusion;
		
	}
	
	
	/**
	* Function importFile_AJAX_get_HeaderMatcher_XHTML()
	*
	* This function is only called via ajax request on the importFile() method. it returns the XHTML
	* for the form calling the function CSVandTableHeaders_Matcher_XHTML() but before
	* it sets the necessary parameters for the CSV with column titles in the first row or not
	* 
	* @param	string 		The file path (absolute path)
	* @param	string 		The table name
	* @param	string 		If CSV has titles in first row ("1" has and "0" does not have)
	*
	* @return	string		The XHTML form for the matcher
	*
	*/
	public function importFile_AJAX_get_HeaderMatcher_XHTML($file, $table, $hasHeaders) {
		
		$this->_currentFile		= $file;
		$this->_tableHeaders	= $this->get_ColumnsNames($table);
		
		if ($hasHeaders == "1") {
			$this->heading 	= true;
			$this->auto($file);
			$rowCount 		= $this->_csvNumLines = count(file($file)) - 1;		// Set number of CSV rows (count total - first row because its headers)
			$colCount 		= $this->processCSV_Titles();						// Set $this->_csvHeaders with CSV column titles (headers)
		} else {
			$this->heading 	= false;
			$this->auto($file);
			$rowCount 		= $this->_csvNumLines = count(file($file));			// Set number of CSV rows (count total - first row because its headers)
			$colCount 		= $this->processCSV_NoTitles();						// Set $this->_csvHeaders with CSV column titles (headers)
		}
		
		return $this->CSVandTableHeaders_Matcher_XHTML($table, true, $rowCount, true);
	}
	
	
	
	
	
	/******************************************************
	   FROM THIS POINT IS FULL CSV VALIDATION FUNCTIONS
	******************************************************/
	
	
	/**
	* Function runFullVerify()
	*
	* This function is only everytime the Full CSV validation is required.
	* It can happen on importFile_Interative() and importFile() methods
	*
	* It goes at every row, counts the total number of values, checks them against the
	* table number of columns. If numbers match, the it goes at every value on that row
	* and check it against the database table column datatype. If everything is ok it carries
	* on to the next row, until it finished the entire CSV file.
	*
	* The full verification also, takes in mind that CSV column title might not match the
	* table column titles, and when that happens it checks for the correct matched csv field
	* for that corresponding table column.
	*
	* The validation will set the erros (if they occur) on the array $this->_FullVerifyERROR
	*
	* Meanwhile, during the process, a progress bar is being flushed to the page. In case the fullverify
	* is called by an ajax request, instead of flushing the progress bar, it increments a session variable
	* using a session write close technic, and that session will be checked by another ajax
	* request to feed in the progress bar. Sweet hun ?
	* 
	* @param	string 		The table name
	* @param	int 		The number of rows in the CSV file
	* @param	bool 		Is this being called by AJAX request						- Optional - Default "false"
	* @param	array		It contains the CSV column manual matches					- Optional - Default "null"
	*
	* @return	array		Return empty array, or in a special case where on any row 
	*						the script detected invalid number of columns, it returns
	*						an array with the bool nbFieldsError true, and the row where it occured
	*
	*/
	private function runFullVerify($table, $rowCount, $isAJAX = false, $AJAX_csvHeaders = null) {
		
		if ((isset($_SESSION["progressbar_value"])) and ($_SESSION["progressbar_value"] == 100))
			unset($_SESSION["progressbar_value"]);
		
		$nbFieldsError	= false;
		// Get all the details about all the columns
		$columnsInfo 	= $this->DBconn->MetaColumns($table);
		// Prepare progress bar, either to be flushed or se the session for AJAX
		if ($this->_outputFVbar === true) {
			if ($isAJAX === false) {
				@apache_setenv('no-gzip', 1);
				@ini_set('zlib.output_compression', 0);
				@ini_set('implicit_flush', 1);
				$ProgressBar = new ProgressBar();
				$ProgressBar->render();
				
			} else {
				$_SESSION["progressbar_value"] = 1;
			}
		}
		
		$r 		= 1;
		// starts looping every single row
		foreach ($this->data as $key => $row):
			$c = 0;
			$num = count($row);
			if ($num != count($this->_csvHeaders)) {
				$nbFieldsError = true;
				@session_start();
				$_SESSION["progressbar_value"] = 100;
				@session_write_close();
				break;
				
			} else {
				if (($isAJAX == true) and ($AJAX_csvHeaders != null)) {
					// here it's because csv column titles DO NOT match the table column titles
					// lets say csv_columnONE goes to tbl_columnONE but csv_columnONE also goes to tbl_columnTWO
					foreach ($AJAX_csvHeaders as $column):
						$value		= $row[$this->_csvHeaders[$column]];
						$max_length	= $columnsInfo[strtoupper($this->_tableHeaders[$c])]->max_length;
						$scale		= $columnsInfo[strtoupper($this->_tableHeaders[$c])]->scale;
						$type		= $columnsInfo[strtoupper($this->_tableHeaders[$c])]->type;
						$not_null	= $columnsInfo[strtoupper($this->_tableHeaders[$c])]->not_null;
						$this->runFullVerify_FIELD($r, $c, $value, $max_length, $scale, $type, $not_null);
						$c++;
					endforeach;
					
				} else {
					// here is normal (direct) import. All csv column titles match the table column titles
					foreach ($row as $value):
						$max_length	= $columnsInfo[strtoupper($this->_tableHeaders[$c])]->max_length;
						$scale		= $columnsInfo[strtoupper($this->_tableHeaders[$c])]->scale;
						$type		= $columnsInfo[strtoupper($this->_tableHeaders[$c])]->type;
						$not_null	= $columnsInfo[strtoupper($this->_tableHeaders[$c])]->not_null;
						$this->runFullVerify_FIELD($r, $c, $value, $max_length, $scale, $type, $not_null);
						$c++;
					endforeach;
				}
			}
			
			if ($this->_outputFVbar === true) {
				if ($isAJAX === false) {
					$ProgressBar->setProgressBarProgress($r*100/$rowCount);
					
				} else {
					@session_start();
					$_SESSION["progressbar_value"] = round($r*100/$rowCount, 0);
					@session_write_close();
					
				}
			}
				
			usleep(125000); 	// sleeps for 0.125 seconds (quarter of a second) on every row
			$r++;
		endforeach;
		
		if ($this->_outputFVbar === true) {
			if ($isAJAX === false) {
				$ProgressBar->setProgressBarProgress(100);
			} else {
				$_SESSION["progressbar_value"] = 100;
			}
		}
		if ($this->_outputFVbar === true)
		
		$output = array();
		$output["nbFieldsError"] 	= $nbFieldsError;
		$output["stoppedatrow"] 	= $r;
		
		return $output;
	}
	
	
	/**
	* Function runFullVerify_FIELD()
	*
	* This function is called once on evry single field of every single row.
	* It's called and passed the appropriate parameters, and this function is
	* the function that actually does the verification of every field.
	*
	* If an error occurs, it calls $this->makeFullVerifyError to add a new error entry to the system.
	* This function returns nothing. It's just meant to build errors (if any)
	* 
	* @param	int 		The actual row number
	* @param	int 		The actual column ID number
	* @param	string 		The actual cell content
	* @param	int			If set, the mysql column max_length
	* @param	int			If set, the mysql column scale value (used for decimal, float, etc, etc)
	* @param	string		The mysql column type value
	* @param	int			If either the column can have a null value (empty) or not (1)
	*
	*/
	private function runFullVerify_FIELD($row, $columnID, $field, $max_length, $scale, $type, $not_null) {
		
		if (substr($type, 0, 3) == "set") {
			$oldType 		= $type;
			$type 			= "set";
			$removechars 	= array("(", ")", "$type", "'");
			$options 		= str_replace($removechars, "", $oldType);
			$options		= explode(",", $options);
			
		} else if (substr($type, 0, 4) == "enum") {
			$oldType 		= $type;
			$type 			= "enum";
			$removechars 	= array("(", ")", "$type", "'");
			$options 		= str_replace($removechars, "", $oldType);
			$options		= explode(",", $options);
		}
		
		
		switch ($type) {
			case "int":
			case "integer":
				if (strlen($field) < 1) {
					if ($not_null == "1")
						$this->makeFullVerifyError(1, $columnID, $row);
				} else {
					if (strlen($field) > $max_length)
						$this->makeFullVerifyError(2, $columnID, $row);
					else if ($this->verify_INT($field) === false)
						$this->makeFullVerifyError(3, $columnID, $row);
				}
				break;
				
				
			case "decimal":
			case "dec":
			case "double":
			case "real":
			case "float":
				if (strlen($field) < 1) {
					if ($not_null == "1")
						$this->makeFullVerifyError(1, $columnID, $row);
				} else {
					if ($this->verify_FLOAT($field) === false)
						$this->makeFullVerifyError(3, $columnID, $row);
					else if ($this->verify_FLOAT_SIZE($field, $max_length, $scale) === false)
						$this->makeFullVerifyError(2, $columnID, $row);
				}
				break;
				
				
			case "enum":
				if (strlen($field) < 1) {
					if ($not_null == "1")
						$this->makeFullVerifyError(1, $columnID, $row);
				} else {
					if ($this->verify_ENUM($field, $options) === false)
						$this->makeFullVerifyError(3, $columnID, $row);
				}
				break;
				
			case "set":
				if (strlen($field) < 1) {
					if ($not_null == "1")
						$this->makeFullVerifyError(1, $columnID, $row);
				} else {
					if ($this->verify_SET($field, $options) === false)
						$this->makeFullVerifyError(3, $columnID, $row);
				}
				break;
				
				
			case "char":
			case "varchar":
			case "tinytext":
			case "text":
			case "mediumtext":
			case "longtext":
				if (strlen($field) < 1) {
					if ($not_null == "1")
						$this->makeFullVerifyError(1, $columnID, $row);
				} else {
					if ($max_length != -1) {
						if (strlen($field) > $max_length)
							$this->makeFullVerifyError(2, $columnID, $row);
					}
				}
				break;
				
				
			case "date":
				if (strlen($field) < 1) {
					if ($not_null == "1")
						$this->makeFullVerifyError(1, $columnID, $row);
				} else {
					if ($this->verify_DATE($field) === false)
						$this->makeFullVerifyError(3, $columnID, $row);
				}
				break;
				
				
			case "datetime":
				if (strlen($field) < 1) {
					if ($not_null == "1")
						$this->makeFullVerifyError(1, $columnID, $row);
				} else {
					if ($this->verify_DATETIME($field) === false)
						$this->makeFullVerifyError(3, $columnID, $row);
				}
				break;
				
				
			case "year":
				if (strlen($field) < 1) {
					if ($not_null == "1")
						$this->makeFullVerifyError(1, $columnID, $row);
				} else {
					if ($this->verify_YEAR($field, $max_length) === false)
						$this->makeFullVerifyError(3, $columnID, $row);
				}
				break;
				
				
			case "timestamp":
				if (strlen($field) < 1) {
					if ($not_null == "1")
						$this->makeFullVerifyError(1, $columnID, $row);
				} else {
					if ($this->verify_TIMESTAMP($field) === false)
						$this->makeFullVerifyError(3, $columnID, $row);
				}
				break;
				
				
			case "time":
				if (strlen($field) < 1) {
					if ($not_null == "1")
						$this->makeFullVerifyError(1, $columnID, $row);
				} else {
					if ($this->verify_TIME($field) === false)
						$this->makeFullVerifyError(3, $columnID, $row);
				}
				break;
			
			
			
			
			
		}
		
	}
	
	
	/**
	* Function verify_SET()
	*
	* This function is called by runFullVerify() above when a validation of a field
	* against the MySQL SET data type is required. Remeber that SET allows
	* multiple options of the valid ones to be inserted in the database.
	* 
	* @param	string 		The string containing the CSV values separated by comma (or single value)
	* @param	array 		The array containing the MySQL table allowed values for this SET data type
	*
	* @return	bool		Return either false for error, or true for field OK
	*
	*/
	public function verify_SET($strArray, $options) {
		$state	= true;
		$fields = explode(",", $strArray);
		foreach($fields as $field) {
			if (!in_array($field, $options)) {
				$state = false;
				break;
			}
		}
		return $state;
	}
	
	
	/**
	* Function verify_ENUM()
	*
	* This function is called by runFullVerify() above when a validation of a field
	* against the MySQL ENUM data type is required. Remeber that ENUM only allows
	* on option of the valid ones to be inserted in the database
	* 
	* @param	string 		The string containing the CSV value
	* @param	array 		The array containing the MySQL table allowed values for this SET data type
	*
	* @return	bool		Return either false for error, or true for field OK
	*
	*/
	public function verify_ENUM($field, $options) {
		if (in_array($field, $options))
			return true;
		else
			return false;
	}
	
	
	/**
	* Function verify_FLOAT()
	*
	* This function is called by runFullVerify() above when a validation of a field
	* against the MySQL FLOAT data type is required.
	* 
	* @param	string 		The field value to test
	*
	* @return	bool		Return either false for error, or true for field OK
	*
	*/
	public function verify_FLOAT($val) {
		if(is_string($val))
			$val = trim($val);
		if ( is_numeric($val) && ( is_float($val) || ( (float) $val > (int) $val || strlen($val) != strlen( (int) $val) ) && (ceil($val)) != 0 ))
			return true;
		else
			return false;
	}
	
	
	/**
	* Function verify_FLOAT_SIZE()
	*
	* This is a special function to double check the float number against
	* the mysql allowed length and scale.
	* 
	* @param	string 		The field value to test
	* @param	int 		The total number of digits allowed (including the scale (scale is the number of decimal places, right of the comma))
	* @param	int 		The number of digits of the scale (decimal places, right of the comma)
	*
	* @return	bool		Return either false for error, or true for field OK
	*
	*/
	public function verify_FLOAT_SIZE($string, $size, $scale) {
		$size = $size - $scale;
		if(!preg_match("/^[-+]?\d{1,$size}(\.\d{0,$scale})?$/",$string)){
			return false;
		} else {
			return true;
		}
	}
	
	
	/**
	* Function verify_INT()
	*
	* This function is called by runFullVerify() above when a validation of a field
	* against the MySQL INTEGER data type is required.
	* 
	* @param	string 		The field value to test
	*
	* @return	bool		Return either false for error, or true for field OK
	*
	*/
	public function verify_INT($int) {
		// First check if it's a numeric value as either a string or number
		if(is_numeric($int) === true) {
			// It's a number, but it has to be an integer
			if((int)$int == $int){
				return true;
			// It's a number, but not an integer, so we fail
			} else {
				return false;
			}
		// Not a number
		} else {
			return false;
		}
	}
	
	
	/**
	* Function verify_DATE()
	*
	* This function is called by runFullVerify() above when a validation of a field
	* against the MySQL DATE data type is required. Remeber that MySQL only accepts
	* dates in the format of YYYY-MM-DD, anything else is invalid
	* 
	* @param	string 		The field value to test
	* @param	string 		The format to test			- Optional - IMPORTANT to always default to "YYYY-MM-DD"
	*
	* @return	bool		Return either false for error, or true for field OK
	*
	*/
	public function verify_DATE($date, $format = 'YYYY-MM-DD') {
		if(strlen($date) >= 8 && strlen($date) <= 10){
			$separator_only = str_replace(array('M','D','Y'),'', $format);
			$separator = $separator_only[0];
			if($separator){
				$regexp = str_replace($separator, "\\" . $separator, $format);
				$regexp = str_replace('MM', '(0[1-9]|1[0-2])', $regexp);
				$regexp = str_replace('M', '(0?[1-9]|1[0-2])', $regexp);
				$regexp = str_replace('DD', '(0[1-9]|[1-2][0-9]|3[0-1])', $regexp);
				$regexp = str_replace('D', '(0?[1-9]|[1-2][0-9]|3[0-1])', $regexp);
				$regexp = str_replace('YYYY', '\d{4}', $regexp);
				$regexp = str_replace('YY', '\d{2}', $regexp);
				if($regexp != $date && preg_match('/'.$regexp.'$/', $date)){
					foreach (array_combine(explode($separator,$format), explode($separator,$date)) as $key=>$value) {
						if ($key == 'YY') $year = '20'.$value;
						if ($key == 'YYYY') $year = $value;
						if ($key[0] == 'M') $month = $value;
						if ($key[0] == 'D') $day = $value;
					}
					if (checkdate($month,$day,$year)) return true;
				}
			}
		}
		return false;
	}
	
	
	/**
	* Function verify_DATETIME()
	*
	* This function is called by runFullVerify() above when a validation of a field
	* against the MySQL DATETIME data type is required.
	* 
	* @param	string 		The field value to test
	*
	* @return	bool		Return either false for error, or true for field OK
	*
	*/
	public function verify_DATETIME($datetime) {
		if (preg_match("/^(\d{4})-(\d{2})-(\d{2}) ([01][0-9]|2[0-3]):([0-5][0-9]):([0-5][0-9])$/", $dateTime, $matches)) {
			if (checkdate($matches[2], $matches[3], $matches[1])) {
				return true;
			}
		}
		return false; 
	}
	
	
	/**
	* Function verify_YEAR()
	*
	* This function is called by runFullVerify() above when a validation of a field
	* against the MySQL YEAR data type is required.
	* 
	* @param	string 		The field value to test
	* @param	string 		The MySQL year type (options are 4 or 2 for either "YYYY" or "YY")
	*
	* @return	bool		Return either false for error, or true for field OK
	*
	*/
	public function verify_YEAR($year, $type) {
		$type = (int)$type;
		if ($type == 4) {
			if ($this->verify_INT($year) === true) {
				$year = (int)$year;
				if (($year > 1900) and ($year < 2156))
					return true;
			}
			return false;
		} else {
			if (strlen($year) == 2) {
				if ($this->verify_INT($year) === true) {
					$year = (int)$year;
					if (($year >= 00) and ($year <= 99))
						return true;
				} 
			}
			return false;
		}
	}
	
	
	/**
	* Function verify_TIMESTAMP()
	*
	* This function is called by runFullVerify() above when a validation of a field
	* against the MySQL TIMESTAMP data type is required.
	* 
	* @param	string 		The field value to test
	*
	* @return	bool		Return either false for error, or true for field OK
	*
	*/
	public function verify_TIMESTAMP($dateToVerify) {
		$MySQL_startDate 	= '1970-01-01 00:00:01';	// Maximun unix timestamp date value
		$MySQL_endDate 		= '2038-01-19 03:14:07';	// Minimum unix timestamp date value
		
		// Let first check if its integer
		if ($this->verify_INT($dateToVerify) === true) {
			// Values conversion
			$MySQL_startDate 	= strtotime($MySQL_startDate);
			$MySQL_endDate 		= strtotime($MySQL_endDate);
			$dateToVerify		= (int)$dateToVerify;
			// Check if given timestamp is between maximun and minimum dates.
			return (($dateToVerify >= $MySQL_startDate) && ($dateToVerify <= $MySQL_endDate));
		} else {
			return false;
		}
	}
	
	
	/**
	* Function verify_TIME()
	*
	* This function is called by runFullVerify() above when a validation of a field
	* against the MySQL TIME data type is required.
	* 
	* @param	string 		The field value to test
	*
	* @return	bool		Return either false for error, or true for field OK
	*
	*/
	public function verify_TIME($time) {
		if ($this->verify_INT($time) === true) {
			$timeLenght		= strlen($time);
			if ($timeLenght <= 2) {
				if (((int)$time < 60) and ((int)$time >= 0))
					return true;
					
			} else if ($timeLenght <= 4) {
				$seconds 	= substr($time, -2);
				$minutes 	= substr($time, 0, $timeLenght - 2);
				if ((((int)$seconds < 60) and ((int)$seconds >= 0)) and (((int)$minutes < 60) and ((int)$minutes >= 0)))
					return true;
					
			} else {
				$seconds 	= substr($time, -2);
				$minutes 	= substr($time, -4, 2);
				$hours 		= substr($time, 0, $timeLenght - 4);
				if ((((int)$seconds < 60) and ((int)$seconds >= 0)) and (((int)$minutes < 60) and ((int)$minutes >= 0)) and (((int)$hours < 839) and ((int)$hours > -839)))
					return true;
			}
		} else {
			if (preg_match("/(?<![0-9:])-?([0-9]|[0-9][0-9]|[0-7][0-9][0-9]|8[0-3][0-8]):([0-5][0-9])(?::([0-5][0-9]))?(?![0-9:])$/", $time, $matches))
				return true;
		}
		return false; 
	}
	
	
	/**
	* Function makeFullVerifyError()
	*
	* This function is called by runFullVerify() after all the test for the specif field are done and
	* an error is found. This functions is responsible for setting the system variable $this->_FullVerifyERROR 
	* with the appropriate generated error, so it can, later, be produced the appropriate XHTML.
	*
	* This function returns nothing. It just builds up the system error array.
	* 
	* @param	int 		The type of the error (1, 2 or 3 for error of field empty and not allowed, Field size invalid, or field not valid for that type)
	* @param	int 		The column id number where the error was found
	* @param	int 		The row number where the error was found
	*
	*/
	private function makeFullVerifyError($type, $columnID, $row) {
		$this->_FullVerifySTATUS = false;
		switch ($type) {
			case 1:
				$this->_FullVerifyERROR[$columnID]["EMPTY"][] = $row;
				break;
			case 2:
				$this->_FullVerifyERROR[$columnID]["SIZE"][] = $row;
				break;
			case 3:
				$this->_FullVerifyERROR[$columnID]["TYPE"][] = $row;
				break;
		}
	}
	
	
	
	
	
	/***********************************************
	   FROM THIS POINT IS XHTML OUTPUT FUNCTIONS
	***********************************************/
	
	
	/**
	* Function importFile_XHTML()
	*
	* This function will create and output the whole structure and forms
	* for the main importFile() method.
	* 
	* @param	array 		The tables to be ignored and not given as choice
	* @param	bool 		If tabel creation is an option
	*
	* @return	string		The XHTML form for the matcher OR any error
	*
	*/
	private function importFile_XHTML($ignoredTables, $allowCreation) {
		
		if ($ignoredTables == "")
			$ignoredTables = array();
		
		$dbTables 	= $this->get_Tables();
		$nbTable	= count($dbTables);
		
		$XHTML	= 	'<div id="IMEXp_OutputContainer" class="IMEXp_mainImport_wrapper">'.
						'<div class="IMEXp_import_step IMEXp_import-selectTable">'.
							'<div class="IMEXp_output_msg IMEXp_output_msg_neutral">'.$this->__L("SYS-SelectTable").'</div>'.
							'<form class="IMEXp_import" name="IMEXp_import-selectTable-form" id="IMEXp_import-selectTable-form">'.
								'<div class="IMEXp_import-lefty">'.
									'<select class="" name="IMEXp_import-selectTable-select" id="IMEXp_import-selectTable-select">';
		if ($allowCreation === true)
			$XHTML	.= 					'<option selected value="NEW">'.$this->__L("SYS-CreateNewTable").'</option>';
			
		for ($a = 0; $a < $nbTable; $a++) {
			if (!in_array($dbTables[$a], $ignoredTables))
				$XHTML	.= 				'<option value="'.$dbTables[$a].'">'.$dbTables[$a].'</option>';
		}
		$XHTML	.= 					'</select>'.
								'</div>'.
								'<div class="IMEXp_import-righty">'.
									'<input type="button" id="IMEXp_import_submit-selectTable" value="'.$this->__L("SYS-NextStep").'" />'.
								'</div>'.
								'<br clear="all" />'.
							'</form>'.
						'</div>'.
						
						'<div class="IMEXp_import_step IMEXp_import_step_disabled IMEXp_import-uploadFile">'.
							'<div class="IMEXp_output_msg IMEXp_output_msg_neutral">'.$this->__L("SYS-UplaodFIle").'</div>'.
							'<form class="IMEXp_import" name="IMEXp_import-uploadFile-form" id="IMEXp_import-uploadFile-form" style="display: none;">'.
								'<div class="IMEXp_import-lefty">'.
									'<input type="file" id="IMEXp_import-uploadFile" />'.
									'<div class="selectedFileUpload"></div>'.
								'</div>'.
								'<div class="IMEXp_import-righty">'.
									'<input type="button" id="IMEXp_import_submit-uploadFile" value="'.$this->__L("SYS-NextStep").'" />'.
								'</div>'.
								'<br clear="all" />'.
								'<div id="IMEXp_upload_statusMsg" class="IMEXp_output_msg IMEXp_output_msg_error" style="display: none;" ></div>'.
							'</form>'.
						'</div>'.
						
						'<div class="IMEXp_import_step IMEXp_import_step_disabled IMEXp_import-csvHeaders">'.
							'<div class="IMEXp_output_msg IMEXp_output_msg_neutral">'.$this->__L("SYS-HasHeaders").'</div>'.
							'<form class="IMEXp_import" name="IMEXp_import-csvHeaders-form" id="IMEXp_import-csvHeaders-form" style="display: none;">'.
								'<div class="IMEXp_import-lefty">'.
									'<input class="IMEXp_import-csvHeaders-check" id="csvHeaders_radio_1" type="radio" name="csvHeaders" value="1" checked />'.
										'<label for="csvHeaders_radio_1" class="radiotitle">'.$this->__L("SYS-YES").'</label>'.
									'<input class="IMEXp_import-csvHeaders-check" id="csvHeaders_radio_2" type="radio" name="csvHeaders" value="0" />'.
										'<label for="csvHeaders_radio_2" class="radiotitle">'.$this->__L("SYS-NO").'</label>'.
								'</div>'.
								'<div class="IMEXp_import-righty">'.
									'<input type="button" id="IMEXp_import_submit-csvHeaders" value="'.$this->__L("SYS-NextStep").'" />'.
								'</div>'.
								'<br clear="all" />'.
							'</form>'.
						'</div>'.
						
						'<div class="IMEXp_import_step IMEXp_import_step_disabled IMEXp_import-csvFullVerify">'.
							'<div class="IMEXp_output_msg IMEXp_output_msg_neutral">'.$this->__L("SYS-PerformFullVer").'</div>'.
							'<form class="IMEXp_import" name="IMEXp_import-csvFullVerify-form" id="IMEXp_import-csvFullVerify-form" style="display: none;">'.
								'<div class="IMEXp_import-lefty">'.
									'<input class="IMEXp_import-fullV-check" id="csvFullVerify_radio_1" type="radio" name="csvFullVerify" value="1" checked />'.
										'<label for="csvFullVerify_radio_1" class="radiotitle">'.$this->__L("SYS-YES").'</label>'.
									'<input class="IMEXp_import-fullV-check" id="csvFullVerify_radio_2" type="radio" name="csvFullVerify" value="0" />'.
										'<label for="csvFullVerify_radio_2" class="radiotitle">'.$this->__L("SYS-NO").'</label>'.
								'</div>'.
								'<div class="IMEXp_import-righty">'.
									'<input type="button" id="IMEXp_import_submit-csvFullVerify" value="'.$this->__L("SYS-NextStep").'" />'.
								'</div>'.
								'<br clear="all" />'.
							'</form>'.
						'</div>'.
						
						'<div class="IMEXp_import_step IMEXp_import_step_disabled IMEXp_import-FinalStep">'.
							'<div class="IMEXp_output_msg IMEXp_output_msg_neutral">'.$this->__L("SYS-ImportProcStart").'</div>'.
							'<form class="IMEXp_import" name="IMEXp_import-FinalStep-form" id="IMEXp_import-FinalStep-form" style="display: none;">'.
								'<div class="IMEXp_import_matcherWrapper"></div>'.
							'</form>'.
						'</div>'.
						
					'</div>'.
					'<div id="IMEXp_import_var_holder" >'.
						'<input type="hidden" id="IMEXp_import_var-selectTable" value="" />'.
						
						'<input type="hidden" id="IMEXp_import_var-uploadFile-status" value="" />'.
						'<input type="hidden" id="IMEXp_import_var-uploadFile-statusMsg" value="" />'.
						'<input type="hidden" id="IMEXp_import_var-uploadFile-file" value="" />'.
						'<input type="hidden" id="IMEXp_import_var-uploadFile-fileName" value="" />'.
						
						'<input type="hidden" id="IMEXp_import_var-csvHeaders" value="" />'.
						'<input type="hidden" id="IMEXp_import_var-csvFullVerify" value="" />'.
					'</div>'.
					'';
		
		return $XHTML;
	}
	
	
	/**
	* Function Create_Table_XHTML()
	*
	* This function creates a form for create table confirmation
	* 
	* @param	string 		The table name
	* @param	bool 		If CSV has titles in first row
	*
	* @return	string		The XHTML form
	*
	*/
	private function Create_Table_XHTML($table, $hasHeaders) {
		
		$this->outputToScreen("warning", $this->__L("SYS-Table") . ' "<strong>'.$table.'</strong>" ' . $this->__L("SYS-notFoundCreate?"), "IMEXp_tablenotfound");
		
		$XHTML	= 	'<div class="IMEXp_OutputContainer_formWrapper">'.
						'<form name="imexp_confirmtablecreation" id="imexp_confirmtablecreation" class="IMEXp_import" >'.
							'<input type="hidden" name="imexp_confirmtablecreation-current_table" id="imexp_confirmtablecreation-current_table" value="'.$table.'" />'.
							'<input type="hidden" name="imexp_confirmtablecreation-current_file" id="imexp_confirmtablecreation-current_file" value="'.addslashes($this->_currentFile).'" />';
		if ($hasHeaders === true)
			$XHTML	.= 		'<input type="hidden" name="imexp_confirmtablecreation-hasHeaders" id="imexp_confirmtablecreation-hasHeaders" value="1" />';
		else
			$XHTML	.= 		'<input type="hidden" name="imexp_confirmtablecreation-hasHeaders" id="imexp_confirmtablecreation-hasHeaders" value="0" />';
		$XHTML	.= 			'<label for="imexp_confirmtablecreation-confirmTableCreation">YES, confirm table confirmation</label> '.
							'<input type="checkbox" name="imexp_confirmtablecreation-confirmTableCreation" id="imexp_confirmtablecreation-confirmTableCreation" value="1" />'.
							'<br clear="all">'.
							'<br clear="all">'.
							'<input type="button" class="submit_import" id="submit_import_imexp_confirmtablecreation" value="'.$this->__L("SYS-SubmitImport").'" />'.
							'<span class="IMEXp_returnMSG"></span>'.
						'</form>'.
					'</div>';
		$output = 	"<script>
						$('#IMEXp_OutputContainer').append('$XHTML');
					</script>
					";
		return $output;
	}
	
	
	
	/**
	* Function CSVandTableHeaders_Matcher_XHTML()
	*
	* This function will check if the number of the columns in the CSV
	* match the number of columns in the table. If not. it return an error,
	* otherwise it creates and returns the Form to match CSV and Table columns
	* 
	* @param	string 		The table name
	* @param	bool 		If full CSV validation is to be made
	* @param	int 		The number os the CSV rows
	* @param	bool		If the request for this form was made VIA AJAX
	*
	* @return	string		The XHTML form for the matcher OR any error
	*
	*/
	private function CSVandTableHeaders_Matcher_XHTML($table, $fullVerify, $numberRows, $isAjax = false) {
		
		$nbCSVheaders 	= count($this->_csvHeaders);
		$nbTABLEheaders = count($this->_tableHeaders);
		
		$tablePK		= $this->DBconn->MetaPrimaryKeys($table); // get table primary key (if any)
		
		if ($nbCSVheaders !== $nbTABLEheaders) {
			
			if ($isAjax === true) {
				$XHTML[]	= 0;
				$XHTML[]	= $this->__L("SYS-NbCSVnoMatchTabl");
			} else {
				$XHTML	= '<div class="IMEXp_output_msg IMEXp_output_msg_error">'.$this->__L("SYS-NbCSVnoMatchTabl").'</div>';
			}
			
		} else {
			
			$XHTML	= 	'<div class="IMEXp_OutputContainer_formWrapper">'.
							'<form name="CSVandTableHeaders_Matcher" id="CSVandTableHeaders_Matcher" class="IMEXp_import" >'.
								'<input type="hidden" name="CSVandTableHeaders_Matcher-number_options" id="CSVandTableHeaders_Matcher-number_options" value="'.$nbTABLEheaders.'" />'.
								'<input type="hidden" name="CSVandTableHeaders_Matcher-current_table" id="CSVandTableHeaders_Matcher-current_table" value="'.$table.'" />'.
								'<input type="hidden" name="CSVandTableHeaders_Matcher-current_file" id="CSVandTableHeaders_Matcher-current_file" value="'.addslashes($this->_currentFile).'" />';							
			if ($fullVerify === true)
				$XHTML	.= 		'<input type="hidden" name="CSVandTableHeaders_Matcher-fullverify" id="CSVandTableHeaders_Matcher-fullverify" value="1" />';
			else
				$XHTML	.= 		'<input type="hidden" name="CSVandTableHeaders_Matcher-fullverify" id="CSVandTableHeaders_Matcher-fullverify" value="0" />';
			$XHTML	.= 			'<input type="hidden" name="CSVandTableHeaders_Matcher-numberRows" id="CSVandTableHeaders_Matcher-numberRows" value="'.$numberRows.'" />'.
								'<span class="PK_info">Note that "<span>'.$tablePK[0].'</span>" is the table primary KEY, used for record updates!</span>'.
								'<div class="tblhead">Table Header Titles</div>'.
								'<div class="csvhead">CSV Header Titles</div>'.
								'<br clear="all" />';
			for ($a = 0; $a < $nbCSVheaders; $a++) {
				$XHTML	.= 		'<select disabled name="table_header_'.$a.'" id="table_header_'.$a.'" >'.
									'<option value="EMPTY">'.$this->__L("SYS-SelectTableHeader").'</option>';
				for ($b = 0; $b < $nbCSVheaders; $b++) {
					if ($b == $a)
						$XHTML	.= 	'<option selected value="'.$b.'">'.$this->_tableHeaders[$b].'</option>';
					else
						$XHTML	.= 	'<option value="'.$b.'">'.$this->_tableHeaders[$b].'</option>';
				}
				$XHTML	.= 		'</select>'.
								'<span class="separator">=</span>'.
								'<select name="csv_header_'.$a.'" id="csv_header_'.$a.'">'.
									'<option value="EMPTY">'.$this->__L("SYS-SelectCSVHeader").'</option>';
				for ($c = 0; $c < $nbCSVheaders; $c++) {
					$XHTML	.= 		'<option value="'.$c.'">'.$this->_csvHeaders[$c].'</option>';
				}
				$XHTML	.= 		'</select><br/>';
			}
			$XHTML	.= 			'<br clear="all" />'.
								'<input type="button" class="submit_import" id="submit_import_CSVandTableHeaders_Matcher" value="'.$this->__L("SYS-SubmitImport").'" />'.
								'<span class="IMEXp_returnMSG"></span>'.
								'<div class="acceptCSVrepeat_container" style="display:none;">'.
									'<label for="CSVandTableHeaders_Matcher-acceptCSVrepeat">YES. Accept repeated CSV fields</label>'.
									'<input type="checkbox" name="CSVandTableHeaders_Matcher-acceptCSVrepeat" id="CSVandTableHeaders_Matcher-acceptCSVrepeat" value="1" />'.
								'</div>'.
							'</form>'.
						'</div>';
		}
		
		if ($isAjax === false) {
			$output = 	"<script>
							$('#IMEXp_OutputContainer').append('$XHTML');
						</script>";
			return $output;
		} else {
			return $XHTML;
			
		}
	}
	
	
	
	
	
	/*************************************************************
	   FROM THIS POINT IS XHTML OUTPUT FUNCTIONS FOR FULLVERIFY
	*************************************************************/
	
	
	/**
	* Function outputFullVerifyError_XHTML()
	*
	* This function creates the necessary XHTML for the CSV Full Validation errors process
	* and outputs the errors according to needs. Two method of output is via PHP flushing
	* the XHTML encolsed in a javascript function so it can be placed properly, or just returning the
	* XHTML if the request is made VIA AJAX. Via ajax only the XHTML is necessary becasue javascript
	* will handle it and place it in place.
	* 
	* @param	bool 		If the request is made after the Colum Titles (headers) matcher form	- OPTIONAL - Default "false"
	* @param	array 		The matched CSV column titles (only comes when the above is true		- OPTIONAL - Default "null"
	* @param	bool		Is an AJAX request, normally true when above is true					- OPTIONAL - Default "false"
	*
	* @return	string		The XHTML for the errors reported
	*
	*/
	private function outputFullVerifyError_XHTML($forHeaderMatch = false, $csv_MatchedHeaders = null, $isAjax = false) {
		$XHTML 	= "<div class=\"IMEXp_FullVerifyReport\"><ul>";
		$num 	= count ($this->_tableHeaders);
		for ($c=0; $c < $num; $c++) {
			$innerOUTPUT 	= "";
			$colName 		= $this->_tableHeaders[$c];
			if (isset($this->_FullVerifyERROR[$c]["EMPTY"])) {
				$innerOUTPUT	.= "<li>";
				$numERR1		= count($this->_FullVerifyERROR[$c]["EMPTY"]);
				if ($numERR1 > 0) {
					if ($forHeaderMatch === true) {
						$CSVcolName  = $this->_csvHeaders[$csv_MatchedHeaders[$c]];
						$innerOUTPUT .= $this->__L("SYS-TheDBcolumn") . " <strong>$colName</strong> (".$this->__L("SYS-MatchedAsCSV")." <strong>$CSVcolName</strong>) ".$this->__L("SYS-hasemptyvalue")." <strong>$numERR1</strong> ".$this->__L("SYS-timesAtRows")." (<span>";
					} else {
						$innerOUTPUT .= $this->__L("SYS-TheColumn") . " <strong>$colName</strong> ".$this->__L("SYS-hasemptyvalue")." <strong>$numERR1</strong> ".$this->__L("SYS-timesAtRows")." (<span>";
					}
					for ($ERR1=0; $ERR1 < $numERR1; $ERR1++) {
						$innerOUTPUT .= $this->_FullVerifyERROR[$c]["EMPTY"][$ERR1] . ", ";
					}
					$innerOUTPUT = substr($innerOUTPUT, 0, -2);
					$innerOUTPUT .= "</span>)";
				}
				$innerOUTPUT .= "</li>";
			}
			
			if (isset($this->_FullVerifyERROR[$c]["SIZE"])) {
				$innerOUTPUT .= "<li>";
				$numERR2 = count($this->_FullVerifyERROR[$c]["SIZE"]);
				if ($numERR2 > 0) {
					if ($forHeaderMatch === true) {
						$CSVcolName  = $this->_csvHeaders[$csv_MatchedHeaders[$c]];
						$innerOUTPUT .= $this->__L("SYS-TheDBcolumn") . " <strong>$colName</strong> (".$this->__L("SYS-MatchedAsCSV")." <strong>$CSVcolName</strong>) ".$this->__L("SYS-haswrongsize")." <strong>$numERR2</strong> ".$this->__L("SYS-timesAtRows")." (<span>";
					} else {
						$innerOUTPUT .= $this->__L("SYS-TheColumn") . " <strong>$colName</strong> ".$this->__L("SYS-haswrongsize")." <strong>$numERR2</strong> ".$this->__L("SYS-timesAtRows")." (<span>";
					}
					for ($ERR2=0; $ERR2 < $numERR2; $ERR2++) {
						$innerOUTPUT .= $this->_FullVerifyERROR[$c]["SIZE"][$ERR2] . ", ";
					}
					$innerOUTPUT = substr($innerOUTPUT, 0, -2);
					$innerOUTPUT .= "</span>)";
				}
				$innerOUTPUT .= "</li>";
			}
			
			if (isset($this->_FullVerifyERROR[$c]["TYPE"])) {
				$innerOUTPUT	.= "<li>";
				$numERR3		= count($this->_FullVerifyERROR[$c]["TYPE"]);
				if ($numERR3 > 0) {
					if ($forHeaderMatch === true) {
						$CSVcolName  = $this->_csvHeaders[$csv_MatchedHeaders[$c]];
						$innerOUTPUT .= $this->__L("SYS-TheDBcolumn") . " <strong>$colName</strong> (".$this->__L("SYS-MatchedAsCSV")." <strong>$CSVcolName</strong>) ".$this->__L("SYS-haswrongtype")." <strong>$numERR3</strong> ".$this->__L("SYS-timesAtRows")." (<span>";
					} else {
						$innerOUTPUT .= $this->__L("SYS-TheColumn") . " <strong>$colName</strong> ".$this->__L("SYS-haswrongtype")." <strong>$numERR3</strong> ".$this->__L("SYS-timesAtRows")." (<span>";
					}
					for ($ERR3=0; $ERR3 < $numERR3; $ERR3++) {
						$innerOUTPUT .= $this->_FullVerifyERROR[$c]["TYPE"][$ERR3] . ", ";
					}
					$innerOUTPUT = substr($innerOUTPUT, 0, -2);
					$innerOUTPUT .= "</span>)";
				}
				$innerOUTPUT .= "</li>";
			}
			
			if ($innerOUTPUT != "")
				$XHTML .= "<li><ul>" . $innerOUTPUT . "</ul></li>";
			
		}
		$XHTML	.= "</ul></div>";
		
		if ($isAjax === false) {
			$output = 	"	<script>
								$('#IMEXp_OutputContainer').append('$XHTML');
							</script>";
			return $output;
		} else {
			return $XHTML;
		}
	}
	
	
	/**
	* Function outputErrorOnLine_XHTML()
	*
	* This function creates the necessary XHTML for the CSV Full Validation specific
	* errors that occur when the Validation Process detects an invlaid number of cells
	* in any given row, according to the number of columns in the destination table.
	* 
	* @param	int 		What row did the problem occured ?
	* @param	bool		Is an AJAX request								- OPTIONAL - Default "false"
	*
	* @return	string		The XHTML for the errors reported
	*
	*/
	private function outputErrorOnLine_XHTML($row, $isAjax = false) {
		if ($this->heading === true)
			$row = $row + 1;
		else
			$row = $row;
		
		$XHTML = 	'<div class="IMEXp_FullVerifyReport">'.
						'<ul>'.
							'<li>'.
								'<ul>'.
									'<li>'.
										$this->__L("SYS-ErrorNumberFields") . " <strong>$row</strong> ".$this->__L("SYS-OfCsvComparedTabl") .
									'</li>'.
								'</ul>'.
							'</li>'.
						'</ul>'.
					'</div>';
		
		if ($isAjax === false) {
			$out = 	"<script>
						$('#IMEXp_OutputContainer').append('$XHTML');
					</script>";
			return $out;
		} else {
			return $XHTML;
		}
	}
	
	
	/**
	* Function start_OutputContainer()
	*
	* This function is only used in the two first methods importFile_Direct() and importFile_Interative()
	* It creates a <div> container so the whole necessary output will be placed.
	*
	* Note that importFile_Interative() is also called middway trough the method importFile() but because
	* that method will be conducted VIA AJAX, when the time in importFile_Interative() comes to call this
	* function it will be ignored, becaused output container has already been done.
	* 
	*
	* @return	string		The XHTML for the DIV container
	*
	*/
	private function start_OutputContainer() {
		echo '<div id="IMEXp_OutputContainer"></div>';
		print str_pad('', intval(ini_get('output_buffering')))."\n";
		@ob_end_flush();
		flush();
	}
	
	
	/**
	* Function outputToScreen()
	*
	* This function is used to flush a message to the above container in the page.
	* 
	* @param	string 		The class for the message, defining Warning, success or error
	* @param	string		The message to outputted to page
	* @param	string		An optional ID for the message									- OPTIONAL - Default ""
	*
	* @return	string		The XHTML for the errors reported
	*
	*/
	private function outputToScreen($class, $string, $id = "") {
		
		if ($this->_outputMsgs === true) {
			echo "
					<script>
						$('#IMEXp_OutputContainer').append(' <div id=\"$id\" class=\"IMEXp_output_msg IMEXp_output_msg_".$class."\">$string</div> ');
					</script>
				";
			print str_pad('', intval(ini_get('output_buffering')))."\n";
			@ob_end_flush();
			flush();
			usleep(1000000*0.5);
		}
	}
	
	
	
	
	
	/*******************************************
	     MISCELANEOUS FUNCTIONS FROM NOW ON
	*******************************************/
	
	
	/**
	* Function processCSV_Titles()
	*
	* This function is to create the CSV Column Titles.
	* It gets the $this->titles from the CSV Parser and allocates
	* them in an array $this->_csvHeaders
	*
	* @return	int		Returns the number of columns in the CSV
	*
	*/
	private function processCSV_Titles() {
		$col 		= 0;
		foreach ($this->titles as $value):
			$this->_csvHeaders[$col] = $value;
			$col++;
		endforeach;
		return $col;
	}
	
	
	/**
	* Function processCSV_NoTitles()
	*
	* This function is to create the CSV Column Titles having
	* in mind that CSV has no Column Titles. So it counts the number of columns
	* and starts building the array $this->_csvHeaders with "Column_1, Column_2, etc"
	*
	* @return	int		Returns the number of columns in the CSV
	*
	*/
	private function processCSV_NoTitles() {
		foreach ($this->data as $key => $row):
			$colCount = count($row);
			break;
		endforeach;
		
		for ($a = 0; $a < $colCount; $a++) {
			$this->_csvHeaders[$a] = "column_" . ($a + 1);
		}
		return $colCount;
	}
	
	
	/**
	* Function get_Tables()
	*
	* This function will retrieve and return all the tables in the database
	*
	* @return	int		Returns an object with all the tables and their properties
	*
	*/
	public function get_Tables() {
		return $this->DBconn->MetaTables();
	}
	
	
	/**
	* Function get_Columns()
	*
	* This function will retrieve all the columns information
	* from the specified table
	*
	* @param	string 		The table name
	*
	* @return	object		Returns an object with all column information (Names, data types, primary keys, etc, etc)
	*
	*/
	public function get_Columns($table) {
		return $this->DBconn->MetaColumns($table);
	}
	
	
	/**
	* Function get_ColumnsNames()
	*
	* This is actually a double function.
	* First it will check if specified table exists, and if true
	* it will retrieve all the column names
	* from the specified table in the database
	*
	* @param	string 		The table name
	*
	* @return	mixed		Returns either false if table does not exist or an object with all column names
	*
	*/
	public function get_ColumnsNames($table) {
		$result = &$this->DBconn->Execute("SHOW TABLES LIKE '".$table."'");
		if ($result->RecordCount() > 0)
			return $this->DBconn->MetaColumnNames($table, true);
		else
			return false;
		
	}
	
	
	/**
	* Function has_dupes()
	*
	* This function simply checks if there
	* are any duplicate values in a given array
	*
	* @param	array 		The array to check for duplicates
	*
	* @return	bool		Returns if there are any duplicates in the array
	*
	*/
	public function has_dupes($array) {
		return count($array) != count(array_unique($array));
	}

	
	/**
	* Function getJSConfig()
	* 
	* This function is only called by Ajax Call from IMEXp_engine.js
	* IMEXp_engine.js will get it's configurations vars from here.
	* NOT ONLY conifg vars, but also any translatable strings. Nice hun ? ;)
	*
	* @return json string 	Returns the json containing all the JS variables
	*
	*/
	public function getJSConfig() {
		return json_encode	( 
								array( 
									// config vars
									'test1' 		=> "KJHKJ", 
									
									// translatable strings for javascript
									'JSLANG'		=> array(
																'TBLVLNB' 			=> $this->__L("JS_TBLVLNB"),
																'CSVVLNB' 			=> $this->__L("JS_CSVVLNB"),
																'ISEMPTY' 			=> $this->__L("JS_ISEMPTY"),
																'Table' 			=> $this->__L("JS-Table"),
																'selected' 			=> $this->__L("JS-selected"),
																'Good' 				=> $this->__L("JS-Good"),
																'TabelCreat' 		=> $this->__L("JS-TabelCreat"),
																'FileNotAllow' 		=> $this->__L("JS-FileNotAllow"),
																'File' 				=> $this->__L("JS-File"),
																'uploaded' 			=> $this->__L("JS-uploaded"),
																'CSVMatch' 			=> $this->__L("JS-CSVMatch"),
																'CSVNOMatch' 		=> $this->__L("JS-CSVNOMatch"),
																'ValidSkip' 		=> $this->__L("JS-ValidSkip"),
																'SUbmitImp' 		=> $this->__L("JS-SUbmitImp"),
																'CSVhasTitle' 		=> $this->__L("JS-CSVhasTitle"),
																'ValidSkipTbl' 		=> $this->__L("JS-ValidSkipTbl"),
																'ImpoProc' 			=> $this->__L("JS-ImpoProc"),
																'STARTED' 			=> $this->__L("JS-STARTED"),
																'SuccessCreatTb' 	=> $this->__L("JS-SuccessCreatTb"),
																'FullVerifyDone' 	=> $this->__L("JS-FullVerifyDone"),
																'ValidSkipOpt' 		=> $this->__L("JS-ValidSkipOpt"),
																'DoMatchNow' 		=> $this->__L("JS-DoMatchNow"),
																'HeadersMatch' 		=> $this->__L("JS-HeadersMatch"),
																'ImportComplet' 	=> $this->__L("JS-ImportComplet"),
																'ImportProcess' 	=> $this->__L("JS-ImportProcess"),
																'HeadersManual' 	=> $this->__L("JS-HeadersManual"),
																'PerforCSV' 		=> $this->__L("JS-PerforCSV"),
																'FullChkDone' 		=> $this->__L("JS-FullChkDone"),
																'NeedConfirmTbl' 	=> $this->__L("JS-NeedConfirmTbl"),
																'SelectFile' 		=> $this->__L("JS-SelectFile")
															)
									
								)
							);
	}
	
	
	/**
	* Function __L()
	* Used to supply a translated string. I know we could get a good gript here using gettext, but thing is...
	* If you are going to include this into a project that already uses gettext, you will be fine converting this function
	* to a gettext equivalent. And this simple function is not about reinventing the wheel. Is just make things easier
	* for a normal end user. Let's be honest, gettext is not that straight forward :)
	* 
	* @param	string 		containing the array index you want the translation
	*
	* @return	string		the translated string value
	*
	*/
	public function __L($phrase){
		// set the global var from the language file
		global $messages;
		
		// now check if language file exists. If not, load default english
		if (file_exists($this->_CFG_langfolder . $this->_CFG_language . ".php"))
			require_once ($this->_CFG_langfolder . $this->_CFG_language . ".php");
		else
			require_once ($this->_CFG_langfolder . "en.php");
		
		// return the desired message
		return $messages[$phrase];
	}
	
	
	
}







/*******************************************
	Extension class for error throwing
*******************************************/

class IMEXp_config_error extends Exception {}









/*********************************************
	FROM NOW ON, IS EXTERNAL CLASSES USED
*********************************************/


/**
 * Progress bar for a lengthy PHP process
 * By Eduardo Pereira | www.voindo.eu
 * Original from http://spidgorny.blogspot.com/2012/02/progress-bar-for-lengthy-php-process.html
 */
 
class ProgressBar {
        var $percentDone = 0;
        var $pbid;
        var $pbarid;
        var $tbarid;
        var $textid;
        var $decimals = 1;
		var $IMEXp;
 
        function __construct($percentDone = 0) {
                $this->pbid = 'pb';
                $this->pbarid = 'progress-bar';
                $this->tbarid = 'transparent-bar';
                $this->textid = 'pb_text';
                $this->percentDone = $percentDone;
				
				$this->IMEXp = new IMEXp();
        }
 
        function render() {
                //print ($GLOBALS['CONTENT']);
                //$GLOBALS['CONTENT'] = '';
                print($this->getContent());
                $this->flush();
                //$this->setProgressBarProgress(0);
        }
 
        function getContent() {
                $this->percentDone = floatval($this->percentDone);
                $percentDone = number_format($this->percentDone, $this->decimals, '.', '') .'%';
				
				if (!isset($content))
					$content = "";
				
                $content .= '<div id="IMEXp_mainProgressDIV">';
					$content .= '<p>'.$this->IMEXp->__L("SYS-PerforCSV").'</p>';
					$content .= '<div id="'.$this->pbid.'" class="pb_container">';
						$content .= '<div id="'.$this->textid.'" class="'.$this->textid.'">'.$percentDone.'</div>';
						$content .= '<div class="pb_bar">';
						$content .= '<div id="'.$this->pbarid.'" class="pb_before" style="width: '.$percentDone.';"></div>';
						$content .= '<div id="'.$this->tbarid.'" class="pb_after"></div>';
					$content .= '</div>';
					$content .= '<br style="height: 1px; font-size: 1px;"/>';
				$content .= '</div>';
				
				$output = "	<script>
								$('#IMEXp_OutputContainer').append('$content');
							</script>";
				
                return $output;
        }
 
        function setProgressBarProgress($percentDone, $text = '') {
                $this->percentDone = $percentDone;
                $text = $text ? $text : number_format($this->percentDone, $this->decimals, '.', '').'%';
                print('
                <script type="text/javascript">
                if (document.getElementById("'.$this->pbarid.'")) {
                        document.getElementById("'.$this->pbarid.'").style.width = "'.$percentDone.'%";');
                if ($percentDone == 100) {
                        print('document.getElementById("'.$this->pbid.'").style.display = "none";');
						print('document.getElementById("IMEXp_mainProgressDIV").style.display = "none";');
                } else {
                        print('document.getElementById("'.$this->tbarid.'").style.width = "'.(100-$percentDone).'%";');
                }
                if ($text) {
                        print('document.getElementById("'.$this->textid.'").innerHTML = "'.htmlspecialchars($text).'";');
                }
                print('}</script>'."\n");
                $this->flush();
        }
 
        function flush() {
                print str_pad('', intval(ini_get('output_buffering')))."\n";
                @ob_end_flush();
                flush();
        }
 
}








class parseCSV {
	
	/*
	Class: parseCSV v0.3.2
	http://code.google.com/p/parsecsv-for-php/
	
	
	Fully conforms to the specifications lined out on wikipedia:
	 - http://en.wikipedia.org/wiki/Comma-separated_values
	
	Based on the concept of Ming Hong Ng's CsvFileParser class:
	 - http://minghong.blogspot.com/2006/07/csv-parser-for-php.html
	
	
	Copyright (c) 2007 Jim Myhrberg (jim@zydev.info).

	Permission is hereby granted, free of charge, to any person obtaining a copy
	of this software and associated documentation files (the "Software"), to deal
	in the Software without restriction, including without limitation the rights
	to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
	copies of the Software, and to permit persons to whom the Software is
	furnished to do so, subject to the following conditions:

	The above copyright notice and this permission notice shall be included in
	all copies or substantial portions of the Software.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
	LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
	THE SOFTWARE.
	
	
	Code Examples
	----------------
	# general usage
	$csv = new parseCSV('data.csv');
	print_r($csv->data);
	----------------
	# tab delimited, and encoding conversion
	$csv = new parseCSV();
	$csv->encoding('UTF-16', 'UTF-8');
	$csv->delimiter = "\t";
	$csv->parse('data.tsv');
	print_r($csv->data);
	----------------
	# auto-detect delimiter character
	$csv = new parseCSV();
	$csv->auto('data.csv');
	print_r($csv->data);
	----------------
	# modify data in a csv file
	$csv = new parseCSV();
	$csv->sort_by = 'id';
	$csv->parse('data.csv');
	# "4" is the value of the "id" column of the CSV row
	$csv->data[4] = array('firstname' => 'John', 'lastname' => 'Doe', 'email' => 'john@doe.com');
	$csv->save();
	----------------
	# add row/entry to end of CSV file
	#  - only recommended when you know the extact sctructure of the file
	$csv = new parseCSV();
	$csv->save('data.csv', array('1986', 'Home', 'Nowhere', ''), true);
	----------------
	# convert 2D array to csv data and send headers
	# to browser to treat output as a file and download it
	$csv = new parseCSV();
	$csv->output (true, 'movies.csv', $array);
	----------------
	*/
	
	
	/**
	* Configuration
	* - set these options with $object->var_name = 'value';
	*/
	
	# use first line/entry as field names
	var $heading = true;
	
	# override field names
	var $fields = array();
	
	# sort entries by this field
	var $sort_by = null;
	var $sort_reverse = false;
	
	# delimiter (comma) and enclosure (double quote)
	var $delimiter = ',';
	var $enclosure = '"';
	
	# basic SQL-like conditions for row matching
	var $conditions = null;
	
	# number of rows to ignore from beginning of data
	var $offset = null;
	
	# limits the number of returned rows to specified amount
	var $limit = null;
	
	# number of rows to analyze when attempting to auto-detect delimiter
	var $auto_depth = 15;
	
	# characters to ignore when attempting to auto-detect delimiter
	var $auto_non_chars = "a-zA-Z0-9\n\r";
	
	# preferred delimiter characters, only used when all filtering method
	# returns multiple possible delimiters (happens very rarely)
	var $auto_preferred = ",;\t.:|";
	
	# character encoding options
	var $convert_encoding = false;
	var $input_encoding = 'ISO-8859-1';
	var $output_encoding = 'ISO-8859-1';
	
	# used by unparse(), save(), and output() functions
	var $linefeed = "\r\n";
	
	# only used by output() function
	var $output_delimiter = ',';
	var $output_filename = 'data.csv';
	
	
	/**
	 * Internal variables
	 */
	
	# current file
	var $file;
	
	# loaded file contents
	var $file_data;
	
	# array of field values in data parsed
	var $titles = array();
	
	# two dimentional array of CSV data
	var $data = array();
	
	
	/**
	 * Constructor
	 * @param   input   CSV file or string
	 * @return  nothing
	 */
	function parseCSV ($input = null, $offset = null, $limit = null, $conditions = null) {
		if ( $offset !== null ) $this->offset = $offset;
		if ( $limit !== null ) $this->limit = $limit;
		if ( count($conditions) > 0 ) $this->conditions = $conditions;
		if ( !empty($input) ) $this->parse($input);
	}
	
	
	// ==============================================
	// ----- [ Main Functions ] ---------------------
	// ==============================================
	
	/**
	 * Parse CSV file or string
	 * @param   input   CSV file or string
	 * @return  nothing
	 */
	function parse ($input = null, $offset = null, $limit = null, $conditions = null) {
		if ( !empty($input) ) {
			if ( $offset !== null ) $this->offset = $offset;
			if ( $limit !== null ) $this->limit = $limit;
			if ( count($conditions) > 0 ) $this->conditions = $conditions;
			if ( is_readable($input) ) {
				$this->data = $this->parse_file($input);
			} else {
				$this->file_data = &$input;
				$this->data = $this->parse_string();
			}
			if ( $this->data === false ) return false;
		}
		return true;
	}
	
	/**
	 * Save changes, or new file and/or data
	 * @param   file     file to save to
	 * @param   data     2D array with data
	 * @param   append   append current data to end of target CSV if exists
	 * @param   fields   field names
	 * @return  true or false
	 */
	function save ($file = null, $data = array(), $append = false, $fields = array()) {
		if ( empty($file) ) $file = &$this->file;
		$mode = ( $append ) ? 'at' : 'wt' ;
		$is_php = ( preg_match('/\.php$/i', $file) ) ? true : false ;
		return $this->_wfile($file, $this->unparse($data, $fields, $append, $is_php), $mode);
	}
	
	/**
	 * Generate CSV based string for output
	 * @param   output      if true, prints headers and strings to browser
	 * @param   filename    filename sent to browser in headers if output is true
	 * @param   data        2D array with data
	 * @param   fields      field names
	 * @param   delimiter   delimiter used to separate data
	 * @return  CSV data using delimiter of choice, or default
	 */
	function output ($output = true, $filename = null, $data = array(), $fields = array(), $delimiter = null) {
		if ( empty($filename) ) $filename = $this->output_filename;
		if ( $delimiter === null ) $delimiter = $this->output_delimiter;
		$data = $this->unparse($data, $fields, null, null, $delimiter);
		if ( $output ) {
			header('Content-type: application/csv');
			header('Content-Disposition: inline; filename="'.$filename.'"');
			echo $data;
		}
		return $data;
	}
	
	/**
	 * Convert character encoding
	 * @param   input    input character encoding, uses default if left blank
	 * @param   output   output character encoding, uses default if left blank
	 * @return  nothing
	 */
	function encoding ($input = null, $output = null) {
		$this->convert_encoding = true;
		if ( $input !== null ) $this->input_encoding = $input;
		if ( $output !== null ) $this->output_encoding = $output;
	}
	
	/**
	 * Auto-Detect Delimiter: Find delimiter by analyzing a specific number of
	 * rows to determine most probable delimiter character
	 * @param   file           local CSV file
	 * @param   parse          true/false parse file directly
	 * @param   search_depth   number of rows to analyze
	 * @param   preferred      preferred delimiter characters
	 * @param   enclosure      enclosure character, default is double quote (").
	 * @return  delimiter character
	 */
	function auto ($file = null, $parse = true, $search_depth = null, $preferred = null, $enclosure = null) {
		
		if ( $file === null ) $file = $this->file;
		if ( empty($search_depth) ) $search_depth = $this->auto_depth;
		if ( $enclosure === null ) $enclosure = $this->enclosure;
		
		if ( $preferred === null ) $preferred = $this->auto_preferred;
		
		if ( empty($this->file_data) ) {
			if ( $this->_check_data($file) ) {
				$data = &$this->file_data;
			} else return false;
		} else {
			$data = &$this->file_data;
		}
		
		$chars = array();
		$strlen = strlen($data);
		$enclosed = false;
		$n = 1;
		$to_end = true;
		
		
		/*
		DO NOT WALK. Theres a bug when the number of columns on a specific row is bigger,
		the parseCSV class returns no CSV headers. Class is not maitained anymore, so this
		workaround works great. No need to find the delimeter. It MUST be a comma (,)
		
		Eduardo - www.voindo.eu
		*/
		/*
		// walk specific depth finding posssible delimiter characters
		for ( $i=0; $i < $strlen; $i++ ) {
			$ch = $data{$i};
			$nch = ( isset($data{$i+1}) ) ? $data{$i+1} : false ;
			$pch = ( isset($data{$i-1}) ) ? $data{$i-1} : false ;
			
			// open and closing quotes
			if ( $ch == $enclosure && (!$enclosed || $nch != $enclosure) ) {
				$enclosed = ( $enclosed ) ? false : true ;
			
			// inline quotes	
			} elseif ( $ch == $enclosure && $enclosed ) {
				$i++;

			// end of row
			} elseif ( ($ch == "\n" && $pch != "\r" || $ch == "\r") && !$enclosed ) {
				if ( $n >= $search_depth ) {
					$strlen = 0;
					$to_end = false;
				} else {
					$n++;
				}
				
			// count character
			} elseif (!$enclosed) {
				if ( !preg_match('/['.preg_quote($this->auto_non_chars, '/').']/i', $ch) ) {
					if ( !isset($chars[$ch][$n]) ) {
						$chars[$ch][$n] = 1;
					} else {
						$chars[$ch][$n]++;
					}
				}
			}
		}
		
		// filtering
		$depth = ( $to_end ) ? $n-1 : $n ;
		$filtered = array();
		foreach( $chars as $char => $value ) {
			if ( $match = $this->_check_count($char, $value, $depth, $preferred) ) {
				$filtered[$match] = $char;
			}
		}
		
		// capture most probable delimiter
		ksort($filtered);
		$delimiter = reset($filtered);
		*/
		$this->delimiter = ",";
		
		// parse data
		if ( $parse ) $this->data = $this->parse_string();
		
		return ",";
		
	}
	
	
	// ==============================================
	// ----- [ Core Functions ] ---------------------
	// ==============================================
	
	/**
	 * Read file to string and call parse_string()
	 * @param   file   local CSV file
	 * @return  2D array with CSV data, or false on failure
	 */
	function parse_file ($file = null) {
		if ( $file === null ) $file = $this->file;
		if ( empty($this->file_data) ) $this->load_data($file);
		return ( !empty($this->file_data) ) ? $this->parse_string() : false ;
	}
	
	/**
	 * Parse CSV strings to arrays
	 * @param   data   CSV string
	 * @return  2D array with CSV data, or false on failure
	 */
	function parse_string ($data = null) {
		if ( empty($data) ) {
			if ( $this->_check_data() ) {
				$data = &$this->file_data;
			} else return false;
		}
		
		$rows = array();
		$row = array();
		$row_count = 0;
		$current = '';
		$head = ( !empty($this->fields) ) ? $this->fields : array() ;
		$col = 0;
		$enclosed = false;
		$was_enclosed = false;
		$strlen = strlen($data);
		
		// walk through each character
		for ( $i=0; $i < $strlen; $i++ ) {
			$ch = $data{$i};
			$nch = ( isset($data{$i+1}) ) ? $data{$i+1} : false ;
			$pch = ( isset($data{$i-1}) ) ? $data{$i-1} : false ;
			
			// open and closing quotes
			if ( $ch == $this->enclosure && (!$enclosed || $nch != $this->enclosure) ) {
				$enclosed = ( $enclosed ) ? false : true ;
				if ( $enclosed ) $was_enclosed = true;
			
			// inline quotes	
			} elseif ( $ch == $this->enclosure && $enclosed ) {
				$current .= $ch;
				$i++;

			// end of field/row
			} elseif ( ($ch == $this->delimiter || ($ch == "\n" && $pch != "\r") || $ch == "\r") && !$enclosed ) {
				if ( !$was_enclosed ) $current = trim($current);
				$key = ( !empty($head[$col]) ) ? $head[$col] : $col ;
				$row[$key] = $current;
				$current = '';
				$col++;
			
				// end of row
				if ( $ch == "\n" || $ch == "\r" ) {
					if ( $this->_validate_offset($row_count) && $this->_validate_row_conditions($row, $this->conditions) ) {
						if ( $this->heading && empty($head) ) {
							$head = $row;
						} elseif ( empty($this->fields) || (!empty($this->fields) && (($this->heading && $row_count > 0) || !$this->heading)) ) {
							if ( !empty($this->sort_by) && !empty($row[$this->sort_by]) ) {
								if ( isset($rows[$row[$this->sort_by]]) ) {
									$rows[$row[$this->sort_by].'_0'] = &$rows[$row[$this->sort_by]];
									unset($rows[$row[$this->sort_by]]);
									for ( $sn=1; isset($rows[$row[$this->sort_by].'_'.$sn]); $sn++ ) {}
									$rows[$row[$this->sort_by].'_'.$sn] = $row;
								} else $rows[$row[$this->sort_by]] = $row;
							} else $rows[] = $row;
						}
					}
					$row = array();
					$col = 0;
					$row_count++;
					if ( $this->sort_by === null && $this->limit !== null && count($rows) == $this->limit ) {
						$i = $strlen;
					}
				}
				
			// append character to current field
			} else {
				$current .= $ch;
			}
		}
		$this->titles = $head;
		if ( !empty($this->sort_by) ) {
			( $this->sort_reverse ) ? krsort($rows) : ksort($rows) ;
			if ( $this->offset !== null || $this->limit !== null ) {
				$rows = array_slice($rows, ($this->offset === null ? 0 : $this->offset) , $this->limit, true);
			}
		}
		return $rows;
	}
	
	/**
	 * Create CSV data from array
	 * @param   data        2D array with data
	 * @param   fields      field names
	 * @param   append      if true, field names will not be output
	 * @param   is_php      if a php die() call should be put on the first
	 *                      line of the file, this is later ignored when read.
	 * @param   delimiter   field delimiter to use
	 * @return  CSV data (text string)
	 */
	function unparse ( $data = array(), $fields = array(), $append = false , $is_php = false, $delimiter = null) {
		if ( !is_array($data) || empty($data) ) $data = &$this->data;
		if ( !is_array($fields) || empty($fields) ) $fields = &$this->titles;
		if ( $delimiter === null ) $delimiter = $this->delimiter;
		
		$string = ( $is_php ) ? "<?php header('Status: 403'); die(' '); ?>".$this->linefeed : '' ;
		$entry = array();
		
		// create heading
		if ( $this->heading && !$append ) {
			foreach( $fields as $key => $value ) {
				$entry[] = $this->_enclose_value($value);
			}
			$string .= implode($delimiter, $entry).$this->linefeed;
			$entry = array();
		}
		
		// create data
		foreach( $data as $key => $row ) {
			foreach( $row as $field => $value ) {
				$entry[] = $this->_enclose_value($value);
			}
			$string .= implode($delimiter, $entry).$this->linefeed;
			$entry = array();
		}
		
		return $string;
	}
	
	/**
	 * Load local file or string
	 * @param   input   local CSV file
	 * @return  true or false
	 */
	function load_data ($input = null) {
		$data = null;
		$file = null;
		if ( $input === null ) {
			$file = $this->file;
		} elseif ( file_exists($input) ) {
			$file = $input;
		} else {
			$data = $input;
		}
		if ( !empty($data) || $data = $this->_rfile($file) ) {
			if ( $this->file != $file ) $this->file = $file;
			if ( preg_match('/\.php$/i', $file) && preg_match('/<\?.*?\?>(.*)/ims', $data, $strip) ) {
				$data = ltrim($strip[1]);
			}
			if ( $this->convert_encoding ) $data = iconv($this->input_encoding, $this->output_encoding, $data);
			if ( substr($data, -1) != "\n" ) $data .= "\n";
			$this->file_data = &$data;
			return true;
		}
		return false;
	}
	
	
	// ==============================================
	// ----- [ Internal Functions ] -----------------
	// ==============================================
	
	/**
	 * Validate a row against specified conditions
	 * @param   row          array with values from a row
	 * @param   conditions   specified conditions that the row must match 
	 * @return  true of false
	 */
	function _validate_row_conditions ($row = array(), $conditions = null) {
		if ( !empty($row) ) {
			if ( !empty($conditions) ) {
				$conditions = (strpos($conditions, ' OR ') !== false) ? explode(' OR ', $conditions) : array($conditions) ;
				$or = '';
				foreach( $conditions as $key => $value ) {
					if ( strpos($value, ' AND ') !== false ) {
						$value = explode(' AND ', $value);
						$and = '';
						foreach( $value as $k => $v ) {
							$and .= $this->_validate_row_condition($row, $v);
						}
						$or .= (strpos($and, '0') !== false) ? '0' : '1' ;
					} else {
						$or .= $this->_validate_row_condition($row, $value);
					}
				}
				return (strpos($or, '1') !== false) ? true : false ;
			}
			return true;
		}
		return false;
	}
	
	/**
	 * Validate a row against a single condition
	 * @param   row          array with values from a row
	 * @param   condition   specified condition that the row must match 
	 * @return  true of false
	 */
	function _validate_row_condition ($row, $condition) {
		$operators = array(
			'=', 'equals', 'is',
			'!=', 'is not',
			'<', 'is less than',
			'>', 'is greater than',
			'<=', 'is less than or equals',
			'>=', 'is greater than or equals',
			'contains',
			'does not contain',
		);
		$operators_regex = array();
		foreach( $operators as $value ) {
			$operators_regex[] = preg_quote($value, '/');
		}
		$operators_regex = implode('|', $operators_regex);
		if ( preg_match('/^(.+) ('.$operators_regex.') (.+)$/i', trim($condition), $capture) ) {
			$field = $capture[1];
			$op = $capture[2];
			$value = $capture[3];
			if ( preg_match('/^([\'\"]{1})(.*)([\'\"]{1})$/i', $value, $capture) ) {
				if ( $capture[1] == $capture[3] ) {
					$value = $capture[2];
					$value = str_replace("\\n", "\n", $value);
					$value = str_replace("\\r", "\r", $value);
					$value = str_replace("\\t", "\t", $value);
					$value = stripslashes($value);
				}
			}
			if ( array_key_exists($field, $row) ) {
				if ( ($op == '=' || $op == 'equals' || $op == 'is') && $row[$field] == $value ) {
					return '1';
				} elseif ( ($op == '!=' || $op == 'is not') && $row[$field] != $value ) {
					return '1';
				} elseif ( ($op == '<' || $op == 'is less than' ) && $row[$field] < $value ) {
					return '1';
				} elseif ( ($op == '>' || $op == 'is greater than') && $row[$field] > $value ) {
					return '1';
				} elseif ( ($op == '<=' || $op == 'is less than or equals' ) && $row[$field] <= $value ) {
					return '1';
				} elseif ( ($op == '>=' || $op == 'is greater than or equals') && $row[$field] >= $value ) {
					return '1';
				} elseif ( $op == 'contains' && preg_match('/'.preg_quote($value, '/').'/i', $row[$field]) ) {
					return '1';
				} elseif ( $op == 'does not contain' && !preg_match('/'.preg_quote($value, '/').'/i', $row[$field]) ) {
					return '1';
				} else {
					return '0';
				}
			}
		}
		return '1';
	}
	
	/**
	 * Validates if the row is within the offset or not if sorting is disabled
	 * @param   current_row   the current row number being processed
	 * @return  true of false
	 */
	function _validate_offset ($current_row) {
		if ( $this->sort_by === null && $this->offset !== null && $current_row < $this->offset ) return false;
		return true;
	}
	
	/**
	 * Enclose values if needed
	 *  - only used by unparse()
	 * @param   value   string to process
	 * @return  Processed value
	 */
	function _enclose_value ($value = null) {
		if ( $value !== null && $value != '' ) {
			$delimiter = preg_quote($this->delimiter, '/');
			$enclosure = preg_quote($this->enclosure, '/');
			if ( preg_match("/".$delimiter."|".$enclosure."|\n|\r/i", $value) || ($value{0} == ' ' || substr($value, -1) == ' ') ) {
				$value = str_replace($this->enclosure, $this->enclosure.$this->enclosure, $value);
				$value = $this->enclosure.$value.$this->enclosure;
			}
		}
		return $value;
	}
	
	/**
	 * Check file data
	 * @param   file   local filename
	 * @return  true or false
	 */
	function _check_data ($file = null) {
		if ( empty($this->file_data) ) {
			if ( $file === null ) $file = $this->file;
			return $this->load_data($file);
		}
		return true;
	}
	
	
	/**
	 * Check if passed info might be delimiter
	 *  - only used by find_delimiter()
	 * @return  special string used for delimiter selection, or false
	 */
	function _check_count ($char, $array, $depth, $preferred) {
		if ( $depth == count($array) ) {
			$first = null;
			$equal = null;
			$almost = false;
			foreach( $array as $key => $value ) {
				if ( $first == null ) {
					$first = $value;
				} elseif ( $value == $first && $equal !== false) {
					$equal = true;
				} elseif ( $value == $first+1 && $equal !== false ) {
					$equal = true;
					$almost = true;
				} else {
					$equal = false;
				}
			}
			if ( $equal ) {
				$match = ( $almost ) ? 2 : 1 ;
				$pref = strpos($preferred, $char);
				$pref = ( $pref !== false ) ? str_pad($pref, 3, '0', STR_PAD_LEFT) : '999' ;
				return $pref.$match.'.'.(99999 - str_pad($first, 5, '0', STR_PAD_LEFT));
			} else return false;
		}
	}
	
	/**
	 * Read local file
	 * @param   file   local filename
	 * @return  Data from file, or false on failure
	 */
	function _rfile ($file = null) {
		if ( is_readable($file) ) {
			if ( !($fh = fopen($file, 'r')) ) return false;
			$data = fread($fh, filesize($file));
			fclose($fh);
			return $data;
		}
		return false;
	}

	/**
	 * Write to local file
	 * @param   file     local filename
	 * @param   string   data to write to file
	 * @param   mode     fopen() mode
	 * @param   lock     flock() mode
	 * @return  true or false
	 */
	function _wfile ($file, $string = '', $mode = 'wb', $lock = 2) {
		if ( $fp = fopen($file, $mode) ) {
			flock($fp, $lock);
			$re = fwrite($fp, $string);
			$re2 = fclose($fp);
			if ( $re != false && $re2 != false ) return true;
		}
		return false;
	}
	
}











/*
 * jQuery File Upload Plugin PHP Class 5.9.1
 * https://github.com/blueimp/jQuery-File-Upload
 *
 * Copyright 2010, Sebastian Tschan
 * https://blueimp.net
 *
 * Licensed under the MIT license:
 * http://www.opensource.org/licenses/MIT
 */

class UploadHandler
{
    protected $options;
    
    function __construct($options=null) {
        $this->options = array(
            'script_url' => $this->getFullUrl().'/',
            'upload_dir' => dirname(__FILE__) . '/files/',
            'upload_url' => $this->getFullUrl().'/files/',
            'param_name' => 'files',
            // Set the following option to 'POST', if your server does not support
            // DELETE requests. This is a parameter sent to the client:
            'delete_type' => 'DELETE',
            // The php.ini settings upload_max_filesize and post_max_size
            // take precedence over the following max_file_size setting:
            'max_file_size' => null,
            'min_file_size' => 1,
            'accept_file_types' => '/(\.|\/)(csv)$/i',
            'max_number_of_files' => null,
            // Set the following option to false to enable resumable uploads:
            'discard_aborted_uploads' => true,
            // Set to true to rotate images based on EXIF meta data, if available:
            'orient_image' => false,
            'image_versions' => array()
        );
        if ($options) {
            $this->options = array_replace_recursive($this->options, $options);
        }
		
		$this->cleanUpFolder(dirname(__FILE__) . '/files/');
		
    }
	
	
	private function cleanUpFolder($DIR) {
		if ($handle = opendir($DIR)) {
			while (false !== ($file = readdir($handle))) {
				if ($file == '.' or $file == '..') continue;
				$filelastmodified = explode("-", $file);
				$filelastmodified = intval($filelastmodified[0]);
				//if(($filelastmodified-time()) > 24*3600) {
				if((time() - $filelastmodified) > 60) {
					unlink($DIR.$file);
				}
			}
			closedir($handle); 
		}
	}
	
	
    protected function getFullUrl() {
      	return
    		(isset($_SERVER['HTTPS']) ? 'https://' : 'http://').
    		(isset($_SERVER['REMOTE_USER']) ? $_SERVER['REMOTE_USER'].'@' : '').
    		(isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : ($_SERVER['SERVER_NAME'].
    		(isset($_SERVER['HTTPS']) && $_SERVER['SERVER_PORT'] === 443 ||
    		$_SERVER['SERVER_PORT'] === 80 ? '' : ':'.$_SERVER['SERVER_PORT']))).
    		substr($_SERVER['SCRIPT_NAME'],0, strrpos($_SERVER['SCRIPT_NAME'], '/'));
    }
    
    protected function set_file_delete_url($file) {
        $file->delete_url = $this->options['script_url']
            .'?file='.rawurlencode($file->fullName);
        $file->delete_type = $this->options['delete_type'];
        if ($file->delete_type !== 'DELETE') {
            $file->delete_url .= '&_method=DELETE';
        }
    }
    
    protected function get_file_object($file_name) {
        $file_path = $this->options['upload_dir'].$file_name;
        if (is_file($file_path) && $file_name[0] !== '.') {
            $file = new stdClass();
            $file->name = $file_name;
            $file->size = filesize($file_path);
            $file->url = $this->options['upload_url'].rawurlencode($file->name);
            foreach($this->options['image_versions'] as $version => $options) {
                if (is_file($options['upload_dir'].$file_name)) {
                    $file->{$version.'_url'} = $options['upload_url']
                        .rawurlencode($file->name);
                }
            }
            $this->set_file_delete_url($file);
            return $file;
        }
        return null;
    }
    
    protected function get_file_objects() {
        return array_values(array_filter(array_map(
            array($this, 'get_file_object'),
            scandir($this->options['upload_dir'])
        )));
    }
    
    protected function has_error($uploaded_file, $file, $error) {
        if ($error) {
            return $error;
        }
        if (!preg_match($this->options['accept_file_types'], $file->name)) {
            return 'acceptFileTypes';
        }
        if ($uploaded_file && is_uploaded_file($uploaded_file)) {
            $file_size = filesize($uploaded_file);
        } else {
            $file_size = $_SERVER['CONTENT_LENGTH'];
        }
        if ($this->options['max_file_size'] && (
                $file_size > $this->options['max_file_size'] ||
                $file->size > $this->options['max_file_size'])
            ) {
            return 'maxFileSize';
        }
        if ($this->options['min_file_size'] &&
            $file_size < $this->options['min_file_size']) {
            return 'minFileSize';
        }
        if (is_int($this->options['max_number_of_files']) && (
                count($this->get_file_objects()) >= $this->options['max_number_of_files'])
            ) {
            return 'maxNumberOfFiles';
        }
        return $error;
    }

    protected function upcount_name_callback($matches) {
        $index = isset($matches[1]) ? intval($matches[1]) + 1 : 1;
        $ext = isset($matches[2]) ? $matches[2] : '';
        return ' ('.$index.')'.$ext;
    }

    protected function upcount_name($name) {
        return preg_replace_callback(
            '/(?:(?: \(([\d]+)\))?(\.[^.]+))?$/',
            array($this, 'upcount_name_callback'),
            $name,
            1
        );
    }
    
    protected function trim_file_name($name, $type) {
        // Remove path information and dots around the filename, to prevent uploading
        // into different directories or replacing hidden system files.
        // Also remove control characters and spaces (\x00..\x20) around the filename:
        $file_name = trim(basename(stripslashes($name)), ".\x00..\x20");
        // Add missing file extension for known image types:
        if (strpos($file_name, '.') === false &&
            preg_match('/^image\/(gif|jpe?g|png)/', $type, $matches)) {
            $file_name .= '.'.$matches[1];
        }
        if ($this->options['discard_aborted_uploads']) {
            while(is_file($this->options['upload_dir'].$file_name)) {
                $file_name = $this->upcount_name($file_name);
            }
        }
        return $file_name;
    }
    
    protected function handle_file_upload($uploaded_file, $name, $size, $type, $error) {
        $file = new stdClass();
        $file->name = $this->trim_file_name($name, $type);
        $file->size = intval($size);
        $file->type = $type;
        $error = $this->has_error($uploaded_file, $file, $error);
        if (!$error && $file->name) {
			@session_start();
			$file->fullName = time() . "-" . md5(session_id()) . "-" . $file->name;
            $file_path = $this->options['upload_dir'] . $file->fullName;
            $append_file = !$this->options['discard_aborted_uploads'] &&
                is_file($file_path) && $file->size > filesize($file_path);
            clearstatcache();
            if ($uploaded_file && is_uploaded_file($uploaded_file)) {
                // multipart/formdata uploads (POST method uploads)
                if ($append_file) {
                    file_put_contents(
                        $file_path,
                        fopen($uploaded_file, 'r'),
                        FILE_APPEND
                    );
                } else {
                    move_uploaded_file($uploaded_file, $file_path);
                }
            } else {
                // Non-multipart uploads (PUT method support)
                file_put_contents(
                    $file_path,
                    fopen('php://input', 'r'),
                    $append_file ? FILE_APPEND : 0
                );
            }
            $file_size = filesize($file_path);
            if ($file_size === $file->size) {
            	if ($this->options['orient_image']) {
            		$this->orient_image($file_path);
            	}
                $file->url = $this->options['upload_url'].rawurlencode($file->name);
                foreach($this->options['image_versions'] as $version => $options) {
                    if ($this->create_scaled_image($file->name, $options)) {
                        if ($this->options['upload_dir'] !== $options['upload_dir']) {
                            $file->{$version.'_url'} = $options['upload_url']
                                .rawurlencode($file->name);
                        } else {
                            clearstatcache();
                            $file_size = filesize($file_path);
                        }
                    }
                }
            } else if ($this->options['discard_aborted_uploads']) {
                unlink($file_path);
                $file->error = 'abort';
            }
            $file->size = $file_size;
            $this->set_file_delete_url($file);
        } else {
            $file->error = $error;
        }
        return $file;
    }
    
    public function get() {
        $file_name = isset($_REQUEST['file']) ?
            basename(stripslashes($_REQUEST['file'])) : null;
        if ($file_name) {
            $info = $this->get_file_object($file_name);
        } else {
            $info = $this->get_file_objects();
        }
        header('Content-type: application/json');
        echo json_encode($info);
    }
    
    public function post() {
        if (isset($_REQUEST['_method']) && $_REQUEST['_method'] === 'DELETE') {
            return $this->delete();
        }
        $upload = isset($_FILES[$this->options['param_name']]) ?
            $_FILES[$this->options['param_name']] : null;
        $info = array();
        if ($upload && is_array($upload['tmp_name'])) {
            // param_name is an array identifier like "files[]",
            // $_FILES is a multi-dimensional array:
            foreach ($upload['tmp_name'] as $index => $value) {
                $info[] = $this->handle_file_upload(
                    $upload['tmp_name'][$index],
                    isset($_SERVER['HTTP_X_FILE_NAME']) ?
                        $_SERVER['HTTP_X_FILE_NAME'] : $upload['name'][$index],
                    isset($_SERVER['HTTP_X_FILE_SIZE']) ?
                        $_SERVER['HTTP_X_FILE_SIZE'] : $upload['size'][$index],
                    isset($_SERVER['HTTP_X_FILE_TYPE']) ?
                        $_SERVER['HTTP_X_FILE_TYPE'] : $upload['type'][$index],
                    $upload['error'][$index]
                );
            }
        } elseif ($upload || isset($_SERVER['HTTP_X_FILE_NAME'])) {
            // param_name is a single object identifier like "file",
            // $_FILES is a one-dimensional array:
            $info[] = $this->handle_file_upload(
                isset($upload['tmp_name']) ? $upload['tmp_name'] : null,
                isset($_SERVER['HTTP_X_FILE_NAME']) ?
                    $_SERVER['HTTP_X_FILE_NAME'] : (isset($upload['name']) ?
                        $upload['name'] : null),
                isset($_SERVER['HTTP_X_FILE_SIZE']) ?
                    $_SERVER['HTTP_X_FILE_SIZE'] : (isset($upload['size']) ?
                        $upload['size'] : null),
                isset($_SERVER['HTTP_X_FILE_TYPE']) ?
                    $_SERVER['HTTP_X_FILE_TYPE'] : (isset($upload['type']) ?
                        $upload['type'] : null),
                isset($upload['error']) ? $upload['error'] : null
            );
        }
        header('Vary: Accept');
        $json = json_encode($info);
        $redirect = isset($_REQUEST['redirect']) ?
            stripslashes($_REQUEST['redirect']) : null;
        if ($redirect) {
            header('Location: '.sprintf($redirect, rawurlencode($json)));
            return;
        }
        if (isset($_SERVER['HTTP_ACCEPT']) &&
            (strpos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false)) {
            header('Content-type: application/json');
        } else {
            header('Content-type: text/plain');
        }
        echo $json;
    }
    
    public function delete() {
        $file_name = isset($_REQUEST['file']) ?
            basename(stripslashes($_REQUEST['file'])) : null;
        $file_path = $this->options['upload_dir'].$file_name;
        $success = is_file($file_path) && $file_name[0] !== '.' && unlink($file_path);
        if ($success) {
            foreach($this->options['image_versions'] as $version => $options) {
                $file = $options['upload_dir'].$file_name;
                if (is_file($file)) {
                    unlink($file);
                }
            }
        }
        header('Content-type: application/json');
        echo json_encode($success);
    }

}











?>